% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/irboost.R
\name{irboost}
\alias{irboost}
\title{fit a robust predictive model with iteratively reweighted boosting algorithm}
\usage{
irboost(
  data,
  label,
  weights,
  params = list(),
  z_init = NULL,
  cfun = "ccave",
  s = 1,
  delta = 0.1,
  iter = 10,
  nrounds = 100,
  del = 1e-10,
  trace = FALSE,
  ...
)
}
\arguments{
\item{data}{input data, if \code{objective="survival:aft"}, it must be an \code{xgb.DMatrix}; otherwise, it can be a matrix of dimension nobs x nvars; each row is an observation vector. Can accept \code{dgCMatrix}}

\item{label}{response variable. Quantitative for \code{objective="reg:squarederror"},\cr 
\code{objective="count:poisson"} (non-negative counts) or \code{objective="reg:gamma"} (positive). 
For \code{objective="binary:logitraw" or "binary:hinge"}, \code{label} should be a factor with two levels}

\item{weights}{vector of nobs with non-negative weights}

\item{params}{the list of parameters, \code{params} is passed to function xgboost::xgboost which requires the same argument. The list must include \code{objective}, a convex component in the CC-family, the second C, or convex down. It is the same as \code{objective} in the \code{xgboost::xgboost}. The following objective functions are currently implemented: 
  \itemize{
    \item \code{reg:squarederror} Regression with squared loss.
    \item \code{binary:logitraw} logistic regression for binary classification, predict linear predictor, not probabilies.
    \item \code{binary:hinge} hinge loss for binary classification. This makes predictions of -1 or 1, rather than   producing probabilities.
    \item \code{multi:softprob} softmax loss function for multiclass problems. The result contains predicted probabilities of each data point in each class, say p_k, k=0, ..., nclass-1. Note, \code{label} is coded as in [0, ..., nclass-1]. The loss function cross-entropy for the i-th observation is computed as -log(p_k) with k=lable_i, i=1, ..., n.
    \item \code{count:poisson}: Poisson regression for count data, predict mean of poisson distribution.
    \item \code{reg:gamma}: gamma regression with log-link, predict mean of gamma distribution. The implementation in \code{xgboost} takes a parameterization in the exponential family:\cr
xgboost/src/src/metric/elementwise_metric.cu.\cr
In particularly, there is only one parameter psi and set to 1. The implementation of the IRCO algorithm follows this parameterization. See Table 2.1, McCullagh and Nelder, Generalized linear models, Chapman & Hall, 1989, second edition.
    \item \code{reg:tweedie}: Tweedie regression with log-link. See also\cr \code{tweedie_variance_power} in range: (1,2). A value close to 2 is like a gamma distribution. A value close to 1 is like a Poisson distribution.
    \item \code{survival:aft}: Accelerated failure time model for censored survival time data. \code{irboost} invokes \code{irb.train_aft}. 
}}

\item{z_init}{vector of nobs with initial convex component values, must be non-negative with default values = weights if provided, otherwise z_init = vector of 1s}

\item{cfun}{concave component of CC-family, can be \code{"hacve", "acave", "bcave", "ccave"}, 
\code{"dcave", "ecave", "gcave", "hcave"}.\cr
See Table 2 at https://arxiv.org/pdf/2010.02848.pdf}

\item{s}{tuning parameter of \code{cfun}. \code{s > 0} and can be equal to 0 for \code{cfun="tcave"}. If \code{s} is too close to 0 for    \code{cfun="acave", "bcave", "ccave"}, the calculated weights can become 0 for all observations, thus crash the program}

\item{delta}{a small positive number provided by user only if \code{cfun="gcave"} and \code{0 < s <1}}

\item{iter}{number of iteration in the IRCO algorithm}

\item{nrounds}{boosting iterations within each IRCO iteration}

\item{del}{convergency criteria in the IRCO algorithm, no relation to \code{delta}}

\item{trace}{if \code{TRUE}, fitting progress is reported}

\item{...}{other arguments passing to \code{xgboost}}
}
\value{
An object with S3 class \code{xgboost} with the additional elments:
\itemize{
  \item \code{weight_update_log} a matrix of \code{nobs} row by \code{iter}      column of observation weights in each iteration of the IRCO algorithm
  \item \code{weight_update} a vector of observation weights in the last IRCO iteration that produces the final model fit
  \item\code{loss_log} sum of loss value of the composite function in each IRCO iteration. Note, \code{cfun} requires \code{objective} non-negative in some cases. Thus care must be taken. For instance, with \code{objective="reg:gamma"}, the loss value is defined by gamma-nloglik - (1+log(min(y))), where y=label. The second term is introduced such that the loss value is non-negative. In fact, gamma-nloglik=y/ypre + log(ypre) in the \code{xgboost}, where ypre is the mean prediction value, can
  be negative. It can be derived that for fixed \code{y}, the minimum value of gamma-nloglik is achived at ypre=y, or 1+log(y). Thus, among all \code{label} values, the minimum of gamma-nloglik is 1+log(min(y)).
}
}
\description{
Fit a predictive model with the iteratively reweighted convex optimization (IRCO) that minimizes the robust loss functions in the CC-family (concave-convex). The convex optimization is conducted by functional descent boosting algorithm in the R package \pkg{xgboost}. The iteratively reweighted boosting (IRBoost) algorithm reduces the weight of the observation that leads to a large loss; it also provides weights to help identify outliers. Applications include the robust generalized
linear models and extensions, where the mean is related to the predictors by boosting, and robust accelerated failure time models.
}
\examples{
\donttest{
# regression, logistic regression, Poisson regression
x <- matrix(rnorm(100*2),100,2)
g2 <- sample(c(0,1),100,replace=TRUE)
fit1 <- irboost(data=x, label=g2, cfun="acave",s=0.5, 
                params=list(objective="reg:squarederror", max_depth=1), trace=TRUE, 
                verbose=0, nrounds=50)
fit2 <- irboost(data=x, label=g2, cfun="acave",s=0.5, 
                params=list(objective="binary:logitraw", max_depth=1), trace=TRUE,  
                verbose=0, nrounds=50)
fit3 <- irboost(data=x, label=g2, cfun="acave",s=0.5, 
                params=list(objective="binary:hinge", max_depth=1), trace=TRUE,  
                verbose=0, nrounds=50)
fit4 <- irboost(data=x, label=g2, cfun="acave",s=0.5, 
                params=list(objective="count:poisson", max_depth=1), trace=TRUE,      
                verbose=0, nrounds=50)

# Gamma regression
x <- matrix(rnorm(100*2),100,2)
g2 <- sample(rgamma(100, 1))
library("xgboost")
param <- list(objective="reg:gamma", max_depth=1)
fit5 <- xgboost(data=x, label=g2, params=param, nrounds=50)
fit6 <- irboost(data=x, label=g2, cfun="acave",s=5, params=param, trace=TRUE, 
                verbose=0, nrounds=50)
plot(predict(fit5, newdata=x), predict(fit6, newdata=x))
hist(fit6$weight_update)
plot(fit6$loss_log)
summary(fit6$weight_update)

# Tweedie regression 
param <- list(objective="reg:tweedie", max_depth=1)
fit6t <- irboost(data=x, label=g2, cfun="acave",s=5, params=param, 
                 trace=TRUE, verbose=0, nrounds=50)
# Gamma vs Tweedie regression
hist(fit6$weight_update)
hist(fit6t$weight_update)
plot(predict(fit6, newdata=x), predict(fit6t, newdata=x))

# multiclass classification in iris dataset:
lb <- as.numeric(iris$Species)-1
num_class <- 3
set.seed(11)

param <- list(objective="multi:softprob", max_depth=4, eta=0.5, nthread=2, 
subsample=0.5, num_class=num_class)
fit7 <- irboost(data=as.matrix(iris[, -5]), label=lb, cfun="acave", s=50,
                params=param, trace=TRUE, verbose=0, nrounds=10)
# predict for softmax returns num_class probability numbers per case:
pred7 <- predict(fit7, newdata=as.matrix(iris[, -5]))
# reshape it to a num_class-columns matrix
pred7 <- matrix(pred7, ncol=num_class, byrow=TRUE)
# convert the probabilities to softmax labels
pred7_labels <- max.col(pred7) - 1
# classification error: 0!
sum(pred7_labels != lb)/length(lb)
table(lb, pred7_labels)
hist(fit7$weight_update)

}
}
\references{
Wang, Zhu (2021), \emph{Unified Robust Boosting}, arXiv eprint, \url{https://arxiv.org/abs/2101.07718}
}
\author{
Zhu Wang\cr Maintainer: Zhu Wang \email{zhuwang@gmail.com}
}
\keyword{classification}
\keyword{regression}
