% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/info.R
\name{info}
\alias{info}
\alias{info.default}
\alias{info.est_item}
\alias{info.est_irt}
\title{Item and Test Information Function}
\usage{
info(x, ...)

\method{info}{default}(x, theta, D = 1, tif = TRUE, ...)

\method{info}{est_item}(x, theta, tif = TRUE, ...)

\method{info}{est_irt}(x, theta, tif = TRUE, ...)
}
\arguments{
\item{x}{A data frame containing item metadata (e.g., item parameters,
number of categories, IRT model types, etc.); or an object of class
\code{est_irt} obtained from \code{\link[=est_irt]{est_irt()}}, or \code{est_item} from
\code{\link[=est_item]{est_item()}}.

See \code{\link[=est_irt]{est_irt()}} or \code{\link[=simdat]{simdat()}} for more details about the item
metadata. This data frame can be easily created using the
\code{\link[=shape_df]{shape_df()}} function.}

\item{...}{Further arguments passed to or from other methods.}

\item{theta}{A numeric vector of theta values at which item and test
information are computed.}

\item{D}{A scaling constant used in IRT models to make the logistic function
closely approximate the normal ogive function. A value of 1.7 is commonly
used for this purpose. Default is 1.}

\item{tif}{Logical. If \code{TRUE}, the test information function
is computed. Default is \code{TRUE}.}
}
\value{
This function returns an object of class \code{info}, which is a list
containing the following components:

\item{iif}{A matrix of item information values. Each row corresponds to an
item, and each column represents the information value computed at a given
theta point. The row names are the item IDs, and the column names indicate
the theta points (e.g., \code{"theta.1"}, \code{"theta.2"}, ...).}

\item{tif}{A numeric vector containing the test information values at each
theta value, computed as the sum of item information values across all
items. This component is included only when \code{tif = TRUE}.}

\item{theta}{A numeric vector of theta values at which the item and test
information functions are evaluated. This matches the user-supplied
\code{theta} argument.}

The returned object is of class \code{info} and can be visualized using
the function \code{\link[=plot.info]{plot.info()}}. This output structure is consistent across
input types (\code{data.frame}, \code{est_item}, \code{est_irt}), and facilitates
downstream plotting, comparison, or export of information function values.
}
\description{
This function computes item and test information functions (Hambleton et
al., 1991) for a given set of theta values.
}
\details{
This function calculates the amount of statistical information provided by
each item (item information function, IIF) and the total test (test
information function, TIF) across a range of ability (theta) values.
Higher information values at a particular theta level indicate greater
measurement precision at that ability level.

The input \code{x} must follow a specific data frame format if not already
an \code{est_irt} or \code{est_item} object. The structure of this data frame
is explained in the documentation of \code{\link[=est_irt]{est_irt()}} and \code{\link[=simdat]{simdat()}}.
Items of different models (e.g., 3PLM, GPCM) can be combined in a single test.

The information is computed for each item appropriately and aggregated for
the TIF if \code{tif = TRUE}. The TIF is often used to assess where the test
provides the most precision, and is critical when designing adaptive tests
or evaluating test coverage across the ability continuum.

The returned object is a list of class \code{"info"}, which contains the item
information matrix and the test information vector. The \code{plot()} method for
\code{info} objects can be used to visualize the IIFs and TIF
(see \code{\link[=plot.info]{plot.info()}}).
}
\section{Methods (by class)}{
\itemize{
\item \code{info(default)}: Default method to compute item and test information functions for
a data frame \code{x} containing the item metadata.

\item \code{info(est_item)}: An object created by the function \code{\link[=est_item]{est_item()}}.

\item \code{info(est_irt)}: An object created by the function \code{\link[=est_irt]{est_irt()}}.

}}
\examples{
## Example 1.
## Using the function `shape_df()` to create a data frame of test metadata

# Create a list of dichotomous item parameters
par.drm <- list(
  a = c(1.1, 1.2, 0.9, 1.8, 1.4),
  b = c(0.1, -1.6, -0.2, 1.0, 1.2),
  g = rep(0.2, 5)
)

# Create a list of polytomous item parameters
par.prm <- list(
  a = c(1.4, 0.6),
  d = list(c(-1.9, 0.0, 1.2), c(0.4, -1.1, 1.5, 0.2))
)

# Create a numeric vector for the number of score categories for each item
cats <- c(2, 4, 2, 2, 5, 2, 2)

# Create a character vector of IRT model types for each item
model <- c("DRM", "GRM", "DRM", "DRM", "GPCM", "DRM", "DRM")

# Create item metadata using `shape_df()`
test <- shape_df(
  par.drm = par.drm, par.prm = par.prm,
  cats = cats, model = model
) # create a data frame

# Define a sequence of theta values
theta <- seq(-2, 2, 0.1)

# Compute item and test information values based on theta
info(x = test, theta = theta, D = 1, tif = TRUE)


## Example 2.
## Using a "-prm.txt" file exported from flexMIRT

# Import a sample "-prm.txt" output file from flexMIRT
flex_prm <- system.file("extdata", "flexmirt_sample-prm.txt",
  package = "irtQ"
)

# Read item parameters and convert them into item metadata
test_flex <- bring.flexmirt(file = flex_prm, "par")$Group1$full_df

# Define a sequence of theta values
theta <- seq(-2, 2, 0.1)

# Compute item and test information values based on theta
info(x = test_flex, theta = theta, D = 1, tif = TRUE)

}
\references{
Hambleton, R. K., & Swaminathan, H. (1985) \emph{Item response theory:
Principles and applications}. Boston, MA: Kluwer.

Hambleton, R. K., Swaminathan, H., & Rogers, H. J. (1991) \emph{Fundamentals of
item response theory}. Newbury Park, CA: Sage.
}
\seealso{
\code{\link[=plot.info]{plot.info()}}, \code{\link[=shape_df]{shape_df()}}, \code{\link[=est_irt]{est_irt()}},
\code{\link[=est_item]{est_item()}}
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
