% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lwrc.R
\name{lwrc}
\alias{lwrc}
\title{Lord-Wingersky Recursion Formula}
\usage{
lwrc(x = NULL, theta, prob = NULL, cats, D = 1)
}
\arguments{
\item{x}{A data frame containing item metadata (e.g., item parameters, number
of categories, IRT model types, etc.). See \code{\link[=est_irt]{est_irt()}} or
\code{\link[=simdat]{simdat()}} for more details about the item metadata. This data frame
can be easily created using the \code{\link[=shape_df]{shape_df()}} function. If \code{prob} is
\code{NULL}, the item metadata in \code{x} is used in the recursion formula. See
\strong{Details} below.}

\item{theta}{A vector of theta values at which the conditional distributions
of observed scores are computed. This argument is required only when item
metadata is provided via the \code{x} argument.}

\item{prob}{A matrix containing the category response probabilities for each
item. Each row corresponds to an item, and each column represents a score
category. If items have different numbers of categories, empty cells should
be filled with zeros or \code{NA} values. If \code{x} is \code{NULL}, this matrix is used
in the recursion formula.}

\item{cats}{A numeric vector specifying the number of score categories for
each item. For example, a dichotomous item has two categories. This
argument is required only when a probability matrix is provided via the
\code{prob} argument.}

\item{D}{A scaling constant used in IRT models to make the logistic function
closely approximate the normal ogive function. A value of 1.7 is commonly
used for this purpose. Default is 1.}
}
\value{
When the \code{prob} argument is provided, the function returns a vector
of probabilities for all possible observed scores on a test.

When the \code{x} argument is specified, it returns a matrix of conditional
probabilities for each observed score across all specified theta values.
}
\description{
This function computes the conditional distributions of number-correct (or
observed) scores given either the probabilities of category responses for
each item or a set of theta values, using the Lord and Wingersky recursion
formula (1984).
}
\details{
The Lord and Wingersky recursive algorithm provides an efficient
method for calculating the compound probabilities of all possible
number-correct (i.e., observed) scores on a test, based on IRT models. This
algorithm is particularly useful for obtaining the IRT model-based
distribution of observed scores.

The conditional distributions of observed scores can be computed using
either the item metadata specified in \code{x} or the category probability
matrix specified in \code{prob}.
}
\examples{
## Example 1: Using a matrix of category probabilities
## This example is from Kolen and Brennan (2004, p. 183)
# Create a matrix of probabilities for correct and incorrect responses to three items
probs <- matrix(c(.74, .73, .82, .26, .27, .18), nrow = 3, ncol = 2, byrow = FALSE)

# Create a vector specifying the number of score categories for each item
cats <- c(2, 2, 2)

# Compute the conditional distribution of observed scores
lwrc(prob = probs, cats = cats)

## Example 2: Using a matrix of category probabilities for a mixed-format test
# Category probabilities for a dichotomous item
p1 <- c(0.2, 0.8, 0, 0, 0)

# Category probabilities for another dichotomous item
p2 <- c(0.4, 0.6, NA, NA, NA)

# Category probabilities for a polytomous item with five categories
p3 <- c(0.1, 0.2, 0.2, 0.4, 0.1)

# Category probabilities for a polytomous item with three categories
p4 <- c(0.5, 0.3, 0.2, NA, NA)

# Combine the probability vectors into a matrix
p <- rbind(p1, p2, p3, p4)

# Create a vector specifying the number of score categories for each item
cats <- c(2, 2, 5, 3)

# Compute the conditional distribution of observed scores
lwrc(prob = p, cats = cats)

## Example 3: Using a data frame of item metadata for a mixed-format test
# Import the "-prm.txt" output file from flexMIRT
flex_prm <- system.file("extdata", "flexmirt_sample-prm.txt", package = "irtQ")

# Read item parameters and convert them to item metadata
x <- bring.flexmirt(file = flex_prm, "par")$Group1$full_df

# Compute the conditional distribution of observed scores for a range of theta values
lwrc(x = x, theta = seq(-4, 4, 0.2), D = 1)

}
\references{
Kolen, M. J. & Brennan, R. L. (2004) \emph{Test Equating, Scaling, and
Linking} (2nd ed.). New York: Springer.

Lord, F. & Wingersky, M. (1984). Comparison of IRT true score and
equipercentile observed score equatings. \emph{Applied Psychological Measurement,
8}(4), 453-461.
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
