#' Obtain the complete list of available data sets (source: I.Stat)
#'
#' @usage list_i_stat(lang = "ita")
#'
#' @param lang language parameter for labels: "ita" for Italian (default), "eng" for English.
#'
#' @return It returns the complete list of available data sets from I.Stat with their ids and names.
#' @export
#'
#' @examples \donttest{list_i_stat()}
list_i_stat <- function(lang = "ita") {
  
  if (!curl::has_internet()) {
    message(
      "No Internet connection available."
    )
    return(invisible(NULL))
  }
  
  istat.fl <- tryCatch(
    rsdmx::readSDMX(providerId = "ISTAT_LEGACY", resource = "dataflow"),
    error = function(e) {
      message(
        "The server is currently unavailable or has changed.\n",
        "Details: ", conditionMessage(e), "\n",
        "Please try again later or use the functions based on the new ISTAT provider (list_istatdata)."
      )
      return(NULL)
    }
  )
  
  if (is.null(istat.fl)) {
    return(invisible(NULL))
  }
  
  istat.fl <- as.data.frame(istat.fl)
  
  ID <- istat.fl$id
  
  if (lang == "ita") {
    if ("Name.it" %in% colnames(istat.fl)) {
      Name <- istat.fl$Name.it
    } else {
      stop("Column `Name.it` not found in the returned dataset.", call. = FALSE)
    }
  } else if (lang == "eng") {
    if ("Name.en" %in% colnames(istat.fl)) {
      Name <- istat.fl$Name.en
    } else {
      stop("Column `Name.en` not found in the returned dataset.", call. = FALSE)
    }
  } else {
    stop("Wrong language parameter. Use 'ita' or 'eng'.", call. = FALSE)
  }
  
  data.frame(ID = ID, Name = Name)
}
