% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jamba-axis.R
\name{minorLogTicksAxis}
\alias{minorLogTicksAxis}
\alias{logFoldAxis}
\alias{pvalueAxis}
\title{Display major and minor tick marks for log-scale axis}
\usage{
minorLogTicksAxis(
  side = NULL,
  lims = NULL,
  logBase = 2,
  displayBase = 10,
  offset = 0,
  symmetricZero = (offset > 0),
  majorCex = 1,
  minorCex = 0.65,
  doMajor = TRUE,
  doMinor = TRUE,
  doLabels = TRUE,
  doMinorLabels = TRUE,
  asValues = TRUE,
  logAxisType = c("normal", "flip", "pvalue"),
  padj = NULL,
  doFormat = TRUE,
  big.mark = ",",
  scipen = 10,
  minorWhich = c(2, 5),
  logStep = 1,
  cex = 1,
  las = 2,
  col = "black",
  col.ticks = col,
  minorLogTicksData = NULL,
  verbose = FALSE,
  ...
)

logFoldAxis(
  side = NULL,
  lims = NULL,
  logBase = 2,
  displayBase = 2,
  offset = 0,
  symmetricZero = TRUE,
  asValues = TRUE,
  minorWhich = NULL,
  doMinor = TRUE,
  doMinorLabels = NULL,
  scipen = 1,
  ...
)

pvalueAxis(
  side = 2,
  lims = NULL,
  displayBase = 10,
  logBase = 10,
  logAxisType = "pvalue",
  asValues = FALSE,
  doMinor = FALSE,
  doMinorLabels = FALSE,
  scipen = 1,
  ...
)
}
\arguments{
\item{side}{\code{integer} indicating the axis side, 1=bottom, 2=left,
3=top, 4=right.}

\item{lims}{NULL or \code{numeric} range for which the axis tick marks
will be determined.
If NULL then the corresponding \code{graphics::par("usr")}
will be used.}

\item{logBase}{\code{numeric} value indicating the log base units, which
will be used similar to how \code{base} is used in \code{log(x, base)}.}

\item{displayBase}{\code{numeric} value indicating the log base units to
use when determining the numeric label position. For example,
data may be log2 scaled, and yet it is visually intuitive to
show log transformed axis units in base 10 units. See examples.}

\item{offset}{\code{numeric} offset used in transforming the
numeric data displayed on this axis. For example, a common
technique is to transform data using \code{log2(1+x)} which adds
\code{1} to values prior to the log2 transformation. In this case,
\code{offset=1}, which ensures the axis labels exactly
match the initial numeric value prior to the log2 transform.}

\item{symmetricZero}{\code{logical} indicating whether numeric values
are symmetric around zero. For example, log fold changes should
use \code{symmetricZero=TRUE} which ensures a log2 value of \code{-2} is
labeled \code{-4} to indicate a negative four fold change. If
\code{symmetricZero=FALSE} a log2 value of \code{-2} would be labeled
\code{0.0625}.}

\item{majorCex, minorCex}{\code{numeric} base text size factors, relative
to cex=1 for default text size. These factors are applied in
addition to existing \code{graphics::par("cex")} values, preserving any
global text size defined there.}

\item{doMajor, doMinor, doLabels, doMinorLabels}{\code{logical}, default TRUE,
whether to display each type of tick and label.
\itemize{
\item \code{doMajor} display major ticks, at \code{displayBase} positions
\item \code{doMinor} display minor ticks at intermediate positions
\item \code{doLabels} display any labels
\item \code{doMinorLabels} display minor labels
}}

\item{asValues}{\code{logical}, default TRUE, whether to print the
exponentiated value, otherwise FALSE will print the log value.}

\item{logAxisType}{\code{character} string with the type of axis values:
\itemize{
\item \code{"normal"}: axis values as-is.
\item \code{"flip"}: inverted axis values, for example where negative values
should be displayed as negative log-transformed values.
\item \code{"pvalue"}: for values transformed as \code{-log10(pvalue)}
}}

\item{padj}{\code{numeric} vector length 2, which is used to position
axis labels for the minor and major labels, respectively. For
example, \code{padj=c(0,1)} will position minor labels just to the
left of the tick marks, and major labels just to the right
of tick marks. This example is helpful when minor labels bunch
up on the right side of each section.}

\item{doFormat}{\code{logical} indicating whether to apply \code{base::format()} to
format numeric labels.}

\item{big.mark, scipen}{arguments passed to \code{base::format()} when
\code{doFormat=TRUE}.}

\item{minorWhich}{\code{integer} vector indicating which of the minor tick
marks should be labeled. Labels are generally numbered from \code{2}
to \code{displayBase-1}. So by default, log 10 units would add
minor tick marks and labels to the \code{c(2,5)} position. For log2
units only, the second label is defined at 1.5, which shows
minor labels at \code{c(3, 6, 12)}, which are \code{1.5 * c(2, 4, 8)}.}

\item{logStep}{\code{integer} the number of log units per "step", typically \code{1}.}

\item{cex, col, col.ticks, las}{parameters used for axis label size,
axis label colors,
axis tick mark colors, and label text orientation, respectively.}

\item{minorLogTicksData}{\code{list} object created by running
\code{jamba::minorLogTicks()}, which allows inspecting and modifying
the content for custom control.}

\item{verbose}{\code{logical} indicating whether to print verbose output.}

\item{...}{Additional arguments are ignored.}
}
\value{
\code{list} with vectors:
\itemize{
\item \code{majorLabels}: \code{character} vector of major axis labels
\item \code{majorTicks}: \code{numeric} vector of major axis tick positions
\item \code{minorLabels}: \code{character} vector of minor axis labels
\item \code{minorTicks}: \code{numeric} vector of minor axis tick positions
\item \code{allLabelsDF}: \code{data.frame} containing all axis tick
positions and corresponding labels.
}
}
\description{
Display major and minor tick marks for log-scale axis,
with optional offset for proper labeling of \code{log2(1+x)}
with numeric offset.

Log fold axis
}
\details{
This function displays log units on the axis of an
existing base R plot. It calls \code{jamba::minorLogTicks()} which
calculates appropriate tick and label positions.

Note: This function assumes the axis values have already been
log-transformed. Make sure to adjust the \code{offset} to reflect
the method of log-transformation, for example:
\itemize{
\item \code{log2(1+x)} would require \code{logBase=2} and \code{offset=1} in order
to represent values properly at or near zero.
\item \code{log(0.5+x)} would require \code{logBase=exp(1)} and \code{offset=0.5}.
\item \code{log10(x)} would require \code{logBase=10} and \code{offset=0}.
}

The defaults \code{logBase=2} and \code{displayBase=10} assume data
has been log2-transformed, and displays tick marks using the
common base of 10. To display tick marks at two-fold intervals,
use \code{displayBase=2}.

This function was motivated in order to label log-transformed
data properly in some special cases, like using \code{log2(1+x)}
where the resulting values are shifted "off by one" using
standard log-scaled axis tick marks and labels.

For log fold changes, set \code{symmetricZero=TRUE}, which will
create negative log scaled fold change values as needed for
negative values. For example, this option would label a
\code{logBase=2} value of \code{-2} as \code{-4} and not as \code{0.25}.

Note that by default, whenever \code{offset > 0} the argument
\code{symmetricZero=TRUE} is also defined, since a negative value in
that scenario has little meaning. This behavior can be turned
off by setting \code{symmetricZero=FALSE}.
}
\examples{
plotPolygonDensity(0:100, breaks=100);

plotPolygonDensity(0:100, breaks=50, log="x",
   main="plotPolygonDensity() uses minorLogTicksAxis()",
   xlab="x (log-scaled)");

plotPolygonDensity(log2(1+0:100), breaks=50,
   main="manually called minorLogTicksAxis(logBase=2)",
   xaxt="n",
   xlab="x (log-scaled)");
minorLogTicksAxis(1, offset=1, logBase=2);

plotPolygonDensity(log10(1+0:100), breaks=50,
   main="manually called minorLogTicksAxis(logBase=10)",
   xaxt="n",
   xlab="x (log-scaled)");
minorLogTicksAxis(1, offset=1, logBase=10);

# example with log fold axes
k <- c(-5:5)
plot(x=k, y=k, xaxt="n", yaxt="n",
   xlab="log2 base, displaying tick marks with log10 intervals",
   ylab="log2 base, displaying tick marks with log2 intervals")
axis(3, las=2)
axis(4, las=2)
lfax <- logFoldAxis(side=1, logBase=2, displayBase=2)
lfay <- logFoldAxis(side=2, logBase=2, displayBase=10)
# optionally add x-axis ablines
abline(v=lfax$allTicks, lty="dotted", col="grey88")
abline(v=lfax$majorTicks, lty="dashed", col="grey82")
# optionally add y-axis ablines
abline(h=lfay$allTicks, lty="dotted", col="grey88")
abline(h=lfay$majorTicks, lty="dashed", col="grey82")

# example showing volcano plot features
set.seed(123);
n <- 1000;
vdf <- data.frame(lfc=rnorm(n) * 2)
vdf$`-log10 (padj)` <- abs(vdf$lfc) * abs(rnorm(n))
plotSmoothScatter(vdf, xaxt="n", yaxt="n", xlab="Fold change",
   main="Volcano plot\ndisplayBase=2")
logFoldAxis(1)
pvalueAxis(2)

plotSmoothScatter(vdf, xaxt="n", yaxt="n", xlab="Fold change",
   main="Volcano plot\ndisplayBase=10")
logFoldAxis(1, displayBase=10)
pvalueAxis(2)
}
\seealso{
Other jam plot functions: 
\code{\link{adjustAxisLabelMargins}()},
\code{\link{coordPresets}()},
\code{\link{decideMfrow}()},
\code{\link{drawLabels}()},
\code{\link{getPlotAspect}()},
\code{\link{groupedAxis}()},
\code{\link{imageByColors}()},
\code{\link{imageDefault}()},
\code{\link{nullPlot}()},
\code{\link{plotPolygonDensity}()},
\code{\link{plotRidges}()},
\code{\link{plotSmoothScatter}()},
\code{\link{shadowText}()},
\code{\link{shadowText_options}()},
\code{\link{showColors}()},
\code{\link{sqrtAxis}()},
\code{\link{usrBox}()}
}
\concept{jam plot functions}
