% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aggregate_omv.R
\name{aggregate_omv}
\alias{aggregate_omv}
\title{Aggregates data from a data set / frame (in long format) and returns them as a .omv-file for the
statistical spreadsheet 'jamovi' (\url{https://www.jamovi.org})}
\usage{
aggregate_omv(
  dtaInp = NULL,
  fleOut = "",
  varAgg = c(),
  grpAgg = c(),
  clcN = FALSE,
  clcMss = FALSE,
  clcMn = FALSE,
  clcMdn = FALSE,
  clcMde = FALSE,
  clcSum = FALSE,
  clcSD = FALSE,
  clcVar = FALSE,
  clcRng = FALSE,
  clcMin = FALSE,
  clcMax = FALSE,
  clcIQR = FALSE,
  drpNA = TRUE,
  usePkg = c("foreign", "haven"),
  selSet = "",
  ...
)
}
\arguments{
\item{dtaInp}{Either a data frame or the name of a data file to be read (including the path,
if required; "FILENAME.ext"; default: NULL); files can be of any supported file
type, see Details below.}

\item{fleOut}{Name of the aggregated data set / file to be written (including the path, if
required; "FILE_OUT.omv"; default: ""); if empty, the resulting data frame is
returned instead.}

\item{varAgg}{A character vector (default: c()) with the names of the variables which shall be
aggregated.}

\item{grpAgg}{A character vector (default: c()) with the variables to group the aggregation
variable (\code{varAgg}) by. If several grouping variables are given, the aggregation
happens for each step / possible combination of these variables. See Details for
more information.}

\item{clcN}{If TRUE, counts the number of valid values for each step / combination of values
in the grouping variable. The suffix appended `"_N"´ is appended to each variable
in the resulting data set.}

\item{clcMss}{If TRUE, counts the number of missing values for each step / combination of values
in the grouping variable. The suffix appended \code{"_Mss"} is appended to each
variable in the resulting data set.}

\item{clcMn}{If TRUE, calculates the mean for each step / combination of values in the grouping
variable. The suffix appended \code{"_Mn"} is appended to each variable in the
resulting data set.}

\item{clcMdn}{If TRUE, calculates the median for each step / combination of values in the
grouping variable. The suffix appended \code{"_Mdn"} is appended to each variable in
the resulting data set.}

\item{clcMde}{If TRUE, calculates the mode for each step / combination of values in the grouping
variable. The suffix appended \code{"_Mde"} is appended to each variable in the
resulting data set.}

\item{clcSum}{If TRUE, calculates the sum for each step / combination of values in the grouping
variable. The suffix appended \code{"_Sum"} is appended to each variable in the
resulting data set.}

\item{clcSD}{If TRUE, calculates the std. deviation for each step / combination of values in
the grouping variable. The suffix appended \code{"_SD"} is appended to each variable in
the resulting data set.}

\item{clcVar}{If TRUE, calculates the variance for each step / combination of values in the
grouping variable. The suffix appended \code{"_Var"} is appended to each variable in
the resulting data set.}

\item{clcRng}{If TRUE, calculates the range for each step / combination of values in the
grouping variable. The suffix appended \code{"_Rng"} is appended to each variable in
the resulting data set.}

\item{clcMin}{If TRUE, determines the minimum at each step / combination of values in the
grouping variable. The suffix appended \code{"_Min"} is appended to each variable in
the resulting data set.}

\item{clcMax}{If TRUE, determines the maximum at each step / combination of values in the
grouping variable. The suffix appended \code{"_Max"} is appended to each variable in
the resulting data set.}

\item{clcIQR}{If TRUE, calculates the IQR for each step / combination of values in the grouping
variable. The suffix appended \code{"_IQR"} is appended to each variable in the
resulting data set.}

\item{drpNA}{If TRUE (default: TRUE), NA values are removed before the aggregation, and the
aggregation is calculated using the valid values. If FALSE, the result would be NA
for any step / combination of values in the grouping variable that contains a NA
value.}

\item{usePkg}{Name of the package: "foreign" or "haven" that shall be used to read SPSS, Stata,
and SAS files; "foreign" is the default (it comes with base R), but "haven" is
newer and more comprehensive.}

\item{selSet}{Name of the data set that is to be selected from the workspace (only applies when
reading .RData-files)}

\item{...}{Additional arguments passed on to methods; see Details below.}
}
\value{
a data frame containing a symmetric matrix (only returned if \code{fleOut} is empty)
containing the distances between the variables / columns (clmDst == TRUE) or rows
(clmDst == FALSE)
}
\description{
Aggregates data from a data set / frame (in long format) and returns them as a .omv-file for the
statistical spreadsheet 'jamovi' (\url{https://www.jamovi.org})
}
\details{
\itemize{
\item \code{varAgg} must be a character vector containing the variables to be aggregated. From these
variables, a new data set is created where the values in the original data set are aggregated
for each possible combination of steps in the grouping variable(s) given in \code{grpAgg}. For
example, if one grouping variable were a participant ID, and another grouping variable were
measurement time points, the resulting new dataset would contain as many rows as the number of
participants multiplied by the number of measurement time points. For each variable in
\code{varAgg}, and each possible aggregation (\code{clcN}, \code{clcMn}, etc. if set to TRUE), a new column
would be generated in the resulting new data set.
\item \code{drpNA} determines whether NA should be dropped / removed before calculating an aggregation.
If set to FALSE, the result for a combination of steps in the grouping variable(s) where at
least one value is NA would be NA. If set to TRUE, only the values for any given combination
of steps in the grouping variable(s) that are not NA are considered for the calculation. For
\code{clcN} and \code{clcMss}, \code{drpNA} has no effect.
NB: If \code{drpNA} is set to TRUE, any row where any of the grouping variable(s) has the value NA
is excluded from the aggregation, if \code{drpNA} is set to FALSE and there is any row where any of
the group variables has the value NA, an error is thrown and no aggregation is carried out.
\item The ellipsis-parameter (\code{...}) can be used to submit arguments / parameters to the functions
that are used for reading and writing the data. By clicking on the respective function under
“See also”, you can get a more detailed overview over which parameters each of those functions
take. The functions are: \code{read_omv} and \code{write_omv} (for jamovi-files), \code{read.table} (for
CSV / TSV files; using similar defaults as \code{read.csv} for CSV and \code{read.delim} for TSV which
both are based upon \code{read.table}), \code{load} (for .RData-files), \code{readRDS} (for .rds-files),
\code{read_sav} (needs the R-package \code{haven}) or \code{read.spss} (needs the R-package \code{foreign}) for
SPSS-files, \code{read_dta} (\code{haven}) / \code{read.dta} (\code{foreign}) for Stata-files, \code{read_sas}
(\code{haven}) for SAS-data-files, and \code{read_xpt} (\code{haven}) / \code{read.xport} (\code{foreign}) for
SAS-transport-files. If you would like to use \code{haven}, you may need to install it using
\code{install.packages("haven", dep = TRUE)}.
}
}
\examples{
# generate a test dataframe with 100 (imaginary) participants / units of
#  observation (ID), 10 measurement (measure) of two variables (V1, V2)
dtaInp <- data.frame(ID = rep(as.character(seq(1, 100)), each = 10),
                     Measure = rep(seq(1, 10), times = 100),
                     V1 = runif(1000,   0, 100),
                     V2 = round(rnorm(1000, 100,  15)))
cat(str(dtaInp))
# the output should look like this
# 'data.frame': 1000 obs. of  4 variables:
#  $ ID     : chr  "1" "1" "1" "1" ...
#  $ Measure: int  1 2 3 4 5 6 7 8 9 10 ...
#  $ V1     : num  ...
#  $ V2     : num  ...
# this data set is stored as (temporary) RDS-file and later processed by long2wide
nmeInp <- tempfile(fileext = ".rds")
nmeOut <- tempfile(fileext = ".omv")
saveRDS(dtaInp, nmeInp)
jmvReadWrite::aggregate_omv(dtaInp = nmeInp, fleOut = nmeOut, varAgg = c("V1", "V2"),
                            grpAgg = "ID", clcN = TRUE, clcMn = TRUE, clcSD = TRUE)
# it is required to give at least the arguments dtaInp, varAgg and grpAgg, each of
# the different switches (clc...) requests a aggregation measure (e.g., mean, median,
# SD, IQR, etc.) to be calculated
# check whether the file was created and its size
cat(list.files(dirname(nmeOut), basename(nmeOut)))
# -> "file[...].omv" ([...] contains a random combination of numbers / characters
cat(file.info(nmeOut)$size)
# -> 4898 (approximate size; size may differ in every run [in dependence of
#          how well the generated random data can be compressed])
cat(str(jmvReadWrite::read_omv(nmeOut, sveAtt = FALSE)))
# the data set contains now the ID variable identifying the different steps of
# aggregation and one column for each combination of aggregation variable (V1 / V2)
# and which calculation was requested (N, mean and SD)
# 'data.frame':	100 obs. of  7 variables:
#  $ ID   : chr  "1" "10" "100" "11" ...
#   ..- attr(*, "jmv-id")= logi TRUE
#   ..- attr(*, "missingValues")= list()
#  $ V1_N : int  10 10 10 10 10 10 10 10 10 10 ...
#   ..- attr(*, "jmv-desc")= chr "V1 (N)"
#   ..- attr(*, "missingValues")= list()
#  $ V1_Mn: num  45.4 51.9 49.4 54.8 47.2 ...
#   ..- attr(*, "jmv-desc")= chr "V1 (Mean)"
#   ..- attr(*, "missingValues")= list()
#  $ V1_SD: num  31.7 31.4 26.5 20.2 29.1 ...
#   ..- attr(*, "jmv-desc")= chr "V1 (SD)"
#   ..- attr(*, "missingValues")= list()
#  $ V2_N : int  10 10 10 10 10 10 10 10 10 10 ...
#   ..- attr(*, "jmv-desc")= chr "V2 (N)"
#   ..- attr(*, "missingValues")= list()
#  $ V2_Mn: num  96.4 102.3 101.6 104.6 108.7 ...
#   ..- attr(*, "jmv-desc")= chr "V2 (Mean)"
#   ..- attr(*, "missingValues")= list()
#  $ V2_SD: num  14.8 18.4 11.2 10.1 14.3 ...
#   ..- attr(*, "jmv-desc")= chr "V2 (SD)"
#   ..- attr(*, "missingValues")= list()

unlink(nmeInp)
unlink(nmeOut)


}
\seealso{
\code{aggregate_omv} internally uses \code{\link[stats:aggregate]{stats::aggregate()}} as function for the aggregation,
and \code{\link[base:NA]{base::is.na()}}, \code{\link[base:mean]{base::mean()}}, \code{\link[stats:median]{stats::median()}}, \code{\link[base:sum]{base::sum()}}, \code{\link[stats:sd]{stats::sd()}},
\code{\link[stats:cor]{stats::var()}}, \code{\link[base:range]{base::range()}}, \code{\link[base:Extremes]{base::min()}}, \code{\link[base:Extremes]{base::max()}}, and
\code{\link[stats:quantile]{stats::quantile()}} for calculation. It furthermore uses the following functions for reading
and writing data files in different formats: \code{\link[=read_omv]{read_omv()}} and
\code{\link[=write_omv]{write_omv()}} for jamovi-files, \code{\link[utils:read.table]{utils::read.table()}} for CSV / TSV files,
\code{\link[=load]{load()}} for reading .RData-files, \code{\link[=readRDS]{readRDS()}} for .rds-files, \code{\link[haven:read_spss]{haven::read_sav()}} or
\code{\link[foreign:read.spss]{foreign::read.spss()}} for SPSS-files, \code{\link[haven:read_dta]{haven::read_dta()}} or \code{\link[foreign:read.dta]{foreign::read.dta()}} for
Stata-files, \code{\link[haven:read_sas]{haven::read_sas()}} for SAS-data-files, and \code{\link[haven:read_xpt]{haven::read_xpt()}} or
\code{\link[foreign:read.xport]{foreign::read.xport()}} for SAS-transport-files.
}
