% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flip2sss.R
\name{flip2sss}
\alias{flip2sss}
\title{flipscores 2-Stage Summary Statistics approach}
\usage{
flip2sss(formula = NULL, data = NULL, cluster = NULL, 
family = "gaussian", summstats_within=NULL, ...)
}
\arguments{
\item{formula}{A formula or a list of formulas. It can be a complete model as.formula or a list of formulas, one for each element produced by the function.}

\item{data}{The dataset to be used for fitting the model.}

\item{cluster}{A vector or a formula evaluated on the data that defines the clusters.}

\item{family}{as in \code{glm}, but given as a character. Not used if argument \code{summstats_within} is not \code{NULL}.}

\item{summstats_within}{A vector of summary statistics model within the data or a function with argument data.}

\item{...}{Other arguments passed to the \code{\link[flipscores]{flipscores}} function.}
}
\value{
A \code{jointest} object, i.e., a list containing the following objects: 
\describe{
 \item{Tspace}{\code{data.frame} where rows represents the sign-flipping transformed (plus the identity one) test and columns the variables.}
 \item{summary_table}{\code{data.frame} containing for each second-step covariate the estimated parameter, score, std error, test , partial correlation and p-value.} 
 \item{mods}{List of \code{glm} objects, i.e., first-step \code{glm} objects}
}
}
\description{
This function fits a model based on the provided formula and data, accounting for clusters and summary statistics within the model.
}
\examples{
library(jointest)
set.seed(123)
# Simulate data
N=20
n=rpois(N,20)
reff=rep(rnorm(N),n)

D=data.frame(X1=rnorm(length(reff)),
             X2=rep(rnorm(N),n),
             Grp=factor(rep(rep(LETTERS[1:3],length.out=N),n)),
             Subj=rep(1:N,n))
D$Y=rbinom(n=nrow(D),prob=plogis( 2*D$X1 * (D$Grp=="B") +  2*D$X2+reff),size=1)

# model of interest formula <- Y ~ Grp * X1 + X2
# clusters structure defined by cluster <- factor(D$Subj)
# The 2-Stage Summary Statistics via flipscore: 
res <- flip2sss(Y ~ Grp * X1 + X2, data=D, 
               cluster=D$Subj, family="binomial")
summary(res)
# This is an ANOVA-like overall test:
summary(combine(res))
# This is an ANOVA-like test:
summary(combine_contrasts(res))

# An alternative and more flexible definition of the model:
# Define the summary statistics (here we propose the glm with firth correction 
# from the logistf package)
summstats_within <- 'logistf::logistf(Y ~ X1, family = binomial(link = "logit"),
control=logistf::logistf.control(maxit=100))'
# however also the classic glm function can be used:
#summstats_within <- 'glm(Y ~ X1, family = binomial(link = "logit"))'

# Then, compute the 2-Stage Summary Statistics approach
# specifying the summary statistics (within cluster/subject)
res <- flip2sss(Y ~ Grp * X1 + X2, data=D, cluster=D$Subj, 
                   summstats_within=summstats_within)
summary(res)

# We can also combine the tests:
# Overall:
summary(combine(res))
# This is similar to an ANOVA test:
summary(combine_contrasts(res))
}
\seealso{
\code{\link{combine_contrasts}}, \code{\link{combine}}
}
\author{
Livio Finos, Angela Andreella
}
