% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compile_keras_grid.R
\name{compile_keras_grid}
\alias{compile_keras_grid}
\title{Compile Keras Models Over a Grid of Hyperparameters}
\usage{
compile_keras_grid(spec, grid, x, y)
}
\arguments{
\item{spec}{A \code{parsnip} model specification created by
\code{create_keras_sequential_spec()} or \code{create_keras_functional_spec()}.}

\item{grid}{A \code{tibble} or \code{data.frame} containing the grid of hyperparameters
to evaluate. Each row represents a unique model architecture to be compiled.}

\item{x}{A data frame or matrix of predictors. This is used to infer the
\code{input_shape} for the Keras model.}

\item{y}{A vector or factor of outcomes. This is used to infer the output
shape and the default loss function for the Keras model.}
}
\value{
A \code{tibble} with the following columns:
\itemize{
\item Columns from the input \code{grid}.
\item \code{compiled_model}: A list-column containing the compiled Keras model
objects. If compilation failed, the element will be \code{NULL}.
\item \code{error}: A list-column containing \code{NA} for successes or a
character string with the error message for failures.
}
}
\description{
Pre-compiles Keras models for each hyperparameter combination in a grid.

This function is a powerful debugging tool to use before running a full
\code{tune::tune_grid()}. It allows you to quickly validate multiple model
architectures, ensuring they can be successfully built and compiled without
the time-consuming process of actually fitting them. It helps catch common
errors like incompatible layer shapes or invalid argument values early.
}
\details{
Compile and Validate Keras Model Architectures

The function iterates through each row of the provided \code{grid}. For each
hyperparameter combination, it attempts to build and compile the Keras model
defined by the \code{spec}. The process is wrapped in a \code{try-catch} block to
gracefully handle and report any errors that occur during model instantiation
or compilation.

The output is a tibble that mirrors the input \code{grid}, with additional columns
containing the compiled model object or the error message, making it easy to
inspect which architectures are valid.
}
\examples{
\donttest{
if (requireNamespace("keras3", quietly = TRUE)) {
library(keras3)
library(parsnip)
library(dials)

# 1. Define layer blocks
input_block <- function(model, input_shape) {
  keras_model_sequential(input_shape = input_shape)
}
hidden_block <- function(model, units = 32) {
  model |> layer_dense(units = units, activation = "relu")
}
output_block <- function(model, num_classes) {
  model |> layer_dense(units = num_classes, activation = "softmax")
}

# 2. Define a kerasnip model specification
create_keras_sequential_spec(
  model_name = "my_mlp_grid",
  layer_blocks = list(
    input = input_block,
    hidden = hidden_block,
    output = output_block
  ),
  mode = "classification"
)

mlp_spec <- my_mlp_grid(
  hidden_units = tune(),
  compile_loss = "categorical_crossentropy",
  compile_optimizer = "adam"
)

# 3. Create a hyperparameter grid
# Include an invalid value (-10) to demonstrate error handling
param_grid <- tibble::tibble(
  hidden_units = c(32, 64, -10)
)

# 4. Prepare dummy data
x_train <- matrix(rnorm(100 * 10), ncol = 10)
y_train <- factor(sample(0:1, 100, replace = TRUE))

# 5. Compile models over the grid
compiled_grid <- compile_keras_grid(
  spec = mlp_spec,
  grid = param_grid,
  x = x_train,
  y = y_train
)

print(compiled_grid)
remove_keras_spec("my_mlp_grid")

# 6. Inspect the results
# The row with `hidden_units = -10` will show an error.
}
}
}
