% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HelperFunctions.R
\name{weibull_shur_correction}
\alias{weibull_shur_correction}
\title{Correction for the Variance-Covariance Matrix for Uncertainty in Scale}
\usage{
weibull_shur_correction(
  X,
  y,
  B,
  dispersion,
  order_list,
  K,
  family,
  observation_weights,
  status
)
}
\arguments{
\item{X}{Block-diagonal matrices of spline expansions}

\item{y}{Block-vector of response}

\item{B}{Block-vector of coefficient estimates}

\item{dispersion}{Scalar, estimate of dispersion, \eqn{ = \text{Weibull scale}^2}}

\item{order_list}{List of partition orders}

\item{K}{Number of partitions minus 1 (\eqn{K})}

\item{family}{Distribution family}

\item{observation_weights}{Optional observation weights (default = 1)}

\item{status}{Censoring indicator (1 = event, 0 = censored)
Indicates whether an event of interest occurred (1) or the observation was
right-censored (0). In survival analysis, right-censoring occurs when the
full survival time is unknown, typically because the study ended or the
subject was lost to follow-up before the event of interest occurred.}
}
\value{
List of \eqn{p \times p} matrices representing the shur-complement corrections \eqn{\textbf{S}_k} to be
elementwise added to each block of the information matrix, before inversion.
}
\description{
Computes the shur complement \eqn{\textbf{S}} such that \eqn{\textbf{G}^* = (\textbf{G}^{-1} + \textbf{S})^{-1}} properly
accounts for uncertainty in estimating dispersion when estimating
variance-covariance. Otherwise, the variance-covariance matrix is optimistic
and assumes the scale is known, when it was in fact estimated. Note that the
parameterization adds the output of this function elementwise (not subtract)
so for most cases, the output of this function will be negative or a
negative definite/semi-definite matrix.
}
\details{
Adjusts the variance-covariance matrix unscaled for coefficients to account
for uncertainty in estimating the Weibull scale parameter, that otherwise
would be lost if simply using \eqn{\textbf{G}=(\textbf{X}^{T}\textbf{W}\textbf{X} + \textbf{L})^{-1}}. This is accomplished
using a correction based on the Shur complement so we avoid having to
construct the entire variance-covariance matrix, or modifying the procedure
for \code{\link{lgspline}} substantially.
For any model with nuisance parameters that must have uncertainty accounted
for, this tool will be helpful.

This both provides a tool for actually fitting Weibull accelerated failure
time (AFT) models, and boilerplate code for users who wish to incorporate
Lagrangian multiplier smoothing splines into their own custom models.
}
\examples{

## Minimal example of fitting a Weibull Accelerated Failure Time model
# Simulating survival data with right-censoring
set.seed(1234)
t1 <- rnorm(1000)
t2 <- rbinom(1000, 1, 0.5)
yraw <- rexp(exp(0.01*t1 + 0.01*t2))
# status: 1 = event occurred, 0 = right-censored
status <- rbinom(1000, 1, 0.25)
yobs <- ifelse(status, runif(1, 0, yraw), yraw)
df <- data.frame(
  y = yobs,
  t1 = t1,
  t2 = t2
)

## Fit model using lgspline with Weibull shur correction
model_fit <- lgspline(y ~ spl(t1) + t2,
                      df,
                      unconstrained_fit_fxn = unconstrained_fit_weibull,
                      family = weibull_family(),
                      need_dispersion_for_estimation = TRUE,
                      dispersion_function = weibull_dispersion_function,
                      glm_weight_function = weibull_glm_weight_function,
                      shur_correction_function = weibull_shur_correction,
                      status = status,
                      opt = FALSE,
                      K = 1)

print(summary(model_fit))

## Fit model using lgspline without Weibull shur correction
naive_fit <- lgspline(y ~ spl(t1) + t2,
                      df,
                      unconstrained_fit_fxn = unconstrained_fit_weibull,
                      family = weibull_family(),
                      need_dispersion_for_estimation = TRUE,
                      dispersion_function = weibull_dispersion_function,
                      glm_weight_function = weibull_glm_weight_function,
                      status = status,
                      opt = FALSE,
                      K = 1)

print(summary(naive_fit))

}
