#' Penalized Logistic Normal Multinomial factor analyzer algorithm
#'
#' Main function that can do PLNM factor analyzer and select the best model based on BIC, AIC or ICL.
#' @param W_count The microbiome count matrix
#'
#' @param range_G All possible number of components. A vector.
#' @param range_Q A specific number of latent dimension.
#' @param model The covaraince structure you choose, there are 2 different models belongs to
#'this family:UUU and GUU. You can choose more than 1 covarance structure to do model selection.
#' @param criteria one of AIC, BIC or ICL. The best model is depends on the criteria you choose. The default is BIC
#' @param range_tuning A range of tuning parameters specified, ranged from 0-1.
#' @param iter Max iterations, default is 150.
#' @param X The regression covariate matrix, which is generated by model.matrix.
#'
#'
#'@return z_ig Estimated latent variable z
#'@return cluster Component labels
#'@return mu_g Estimated component mean
#'@return pi_g Estimated component proportion
#'@return B_g Estimated bicluster membership
#'@return D_g Estimated error covariance
#'@return COV Estimated component covariance
#'@return beta_g Estimated covariate coefficients
#'@return overall_loglik Complete log likelihood value for each iteration
#'@return ICL ICL value
#'@return BIC BIC value
#'@return AIC AIC value
#'@return all_fitted_model display all names of fitted models in a data.frame.
#'
#'
#'@examples
#'#'#generate toy data with n=100, K=5,
#'#set up parameters
#'n<-100
#'p<-5
#'mu1<-c(-2.8,-1.3,-1.6,-3.9,-2.6)
#'B1<-matrix(c(1,0,1,0,1,0,0,1,0,1),nrow = p, byrow=TRUE)
#'T1<-diag(c(2.9,0.5))
#'D1<-diag(c(0.52, 1.53, 0.56, 0.19, 1.32))
#'cov1<-B1%*%T1%*%t(B1)+D1
#'mu2<-c(1.5,-2.7,-1.1,-0.4,-1.4)
#'B2<-matrix(c(1,0,1,0,0,1,0,1,0,1),nrow = p, byrow=TRUE)
#'T2<-diag(c(0.2,0.003))
#'D2<-diag(c(0.01, 0.62, 0.45, 0.01, 0.37))
#'cov2<-B2%*%T2%*%t(B2)+D2
#'
#'#generate normal distribution
#'library(mvtnorm)
#'simp<-rmultinom(n,1,c(0.6,0.4))
#'lab<-as.factor(apply(t(simp),1,which.max))
#'df<-matrix(0,nrow=n,ncol=p)
#'for (i in 1:n) {
#'  if(lab[i]==1){df[i,]<-rmvnorm(1,mu1,sigma = cov1)}
#'  else if(lab[i]==2){df[i,]<-rmvnorm(1,mu2,sigma = cov2)}
#'}
#'#apply inverse of additive log ratio and transform normal to count data
#'f_df<-cbind(df,0)
#'z<-exp(f_df)/rowSums(exp(f_df))
#'W_count<-matrix(0,nrow=n,ncol=p+1)
#'for (i in 1:n) {
#'  W_count[i,]<-rmultinom(1,runif(1,10000,20000),z[i,])
#'}
#'
#'#if run one model let range_G, and range_tuning be an integer
#'#remember you can always overspecify Q, so we don't suggest to run models with a range of Q.
#'\donttest{res<-plnmfa(W_count,2,2,model="UUU",range_tuning=0.6)}
#'
#'#if run model selection let any \code{range_} parameters be a vector.
#'\donttest{res<-plnmfa(W_count,c(2:3),3,range_tuning=seq(0.5,0.8,by=0.1))}
#'
#'
#'
#'
#'
#'
#' @export
plnmfa <-
  function(W_count,
           range_G,
           range_Q,
           model,
           criteria,
           range_tuning,
           iter,X) {

    allmodels=c("UUU","GUU")
    #default for model.matrix X
    if(missing(X)){
      X<-matrix(rep(1,dim(W_count)[1]),ncol=1)
    }else{X<-X}


    #default for model to select
    if (missing(model)) {
      model <- allmodels
    }
    else if(all(model%in%allmodels))
    {
      model <- model
    }
    else{
      return("Please don't fit models other than the suggestions.")
    }


    #default for criteria
    if (missing(criteria)) {
      criteria <- "BIC"
    } else{
      criteria <- criteria
    }


    #default for iteration
    if (missing(iter)) {
      iter = 150
    } else{
      iter = iter
    }
    #calculate constant permutation term
    const <- NULL
    for (i in 1:dim(W_count)[1]) {
      const[i]=constant_fun(W_count[i,],dim(W_count)[2]-1)
    }



    if (length(range_G) == 1 & length(range_tuning) == 1 &
        length(model) == 1) {
      message(paste(
        c(
          "Best model ",
          model,
          " G = ",
          range_G,
          " Q= ",
          range_Q,
          " T=",
          range_tuning
        ),
        collapse = ""
      ))
      initial_guess <-
        try(initial_variational_lasso(W_count, range_G, range_Q, model,X))
      V <- initial_guess$new_V
      m <- initial_guess$new_m
      pi_g <- initial_guess$new_pi_g
      mu_g <- initial_guess$new_mu_g
      sig_g <- initial_guess$new_sig_g
      B_K <- initial_guess$new_B_g
      T_K <- initial_guess$new_T_g
      D_K <- initial_guess$new_D_g
      beta_g<-initial_guess$new_beta_g
      b_t <- rep(range_tuning, range_G)
      res <-
        try(Mico_bi_lasso(W_count,
                          range_G,
                          range_Q,
                          pi_g,
                          mu_g,
                          sig_g,
                          V,
                          m,
                          B_K,
                          T_K,
                          D_K,
                          model,
                          b_t,
                          iter, const,beta_g,X))
      return(res)

    } else{
      iniG <- model_selection_PGMM(W_count, range_G, range_Q, model,permutation=FALSE,iter, const,X)
      iniG <- iniG[order(iniG[, names(iniG) == criteria], decreasing = T), ]

      best <- iniG[which.max(iniG[, names(iniG) == criteria]), 1]
      best_split <- stringr::str_split(best, " ")

      #get best model and G
      b_cov <- best_split[[1]][1]
      b_G <- as.numeric(stringr::str_remove(best_split[[1]][2], "G"))
      b_Q <- as.numeric(stringr::str_split(best_split[[1]][3], "")[[1]][-1])


      all <-
        model_selection_lasso(W_count, b_G, b_Q, b_cov, range_tuning, iter, const,X)
      all <-
        all[order(all[, names(all) == criteria], decreasing = T),]

      best <- all[which.max(all[, names(all) == criteria]), 1]
      best_split <- stringr::str_split(best, " ")

      #get best tuning parameter
      b_t <- as.numeric(stringr::str_remove(best_split[[1]][4], "T"))



      message(paste(
        c("Best model ", b_cov, " G = ", b_G, " Q= ", b_Q, " T=", b_t[1]),
        collapse = ""
      ))

      initial_guess <-
        try(initial_variational_lasso(W_count, b_G, b_Q, b_cov,X))
      V <- initial_guess$new_V
      m <- initial_guess$new_m
      pi_g <- initial_guess$new_pi_g
      mu_g <- initial_guess$new_mu_g
      sig_g <- initial_guess$new_sig_g
      B_K <- initial_guess$new_B_g
      T_K <- initial_guess$new_T_g
      D_K <- initial_guess$new_D_g
      beta_g<-initial_guess$new_beta_g
      b_t <- rep(b_t, b_G)
      res <-
        try(Mico_bi_lasso(W_count,
                          b_G,
                          b_Q,
                          pi_g,
                          mu_g,
                          sig_g,
                          V,
                          m,
                          B_K,
                          T_K,
                          D_K,
                          b_cov,
                          b_t,
                          iter, const,beta_g,X))
      return(list(best_model = res, all_fitted_model = all))
    }

  }
