% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/monad.R
\name{monad}
\alias{monad}
\alias{\%>>\%}
\alias{\%>-\%}
\alias{fmap}
\alias{bind}
\alias{join}
\title{Monad Operators and Generics}
\usage{
lhs \%>>\% rhs

lhs \%>-\% rhs

fmap(m, f, ...)

bind(m, f, ...)

join(m)
}
\arguments{
\item{m, lhs}{A monadic object.}

\item{f, rhs}{A function. For \code{bind()}, it should return a monadic object.}

\item{...}{Additional arguments passed to \code{f}.}
}
\value{
A monadic object.
}
\description{
Classes implementing methods for these S7 generics are called monads.
\code{fmap()} should be implemented such that the \link[=functor-laws]{functor
laws} hold. \code{bind()} or \code{join()} should be implemented such that the
\link[=monad-laws]{monad laws} hold. \verb{\%>>\%} is the \code{fmap()} pipe operator,
and \verb{\%>-\%} is the \code{bind()} pipe operator. Operator usage is in the form \code{m \%>>\% f(...)}.
}
\section{Details}{


Monads are containers for values. \code{fmap()} transforms the contained value
with a function. \code{bind()} transforms the contained value with a function
that returns a monadic object. \code{join()} takes a monad whose contained value
is another monad, and combines them into a new monadic object. It's used to
unwrap a layer of monadic structure. Implementing classes typically embed
some form of control flow or state management in \code{bind()} or \code{join()}.

There's a default implementation for \code{join()} if you provide \code{bind()}, and
there's a default implementation for \code{bind()} if you provide \code{join()} and
\code{fmap()}. For performance reasons you may wish to implement both
regardless.
}

\section{Operators}{


The pipe operators expect a monadic object as \code{lhs} and a function or a
call expression as \code{rhs}. A call in \code{rhs} is treated as partial application
of the function \code{f}. The pipe expression is transformed into a call to the
corresponding monad generic with any call arguments in \code{rhs} passed as
additional arguments to \code{f} in the generic. For example, \code{m \%>>\% f(x)} is
equivalent to \code{fmap(m, f, x)} and \code{m \%>-\% f(x)} is equivalent to \code{bind(m, f, x)}.
}

\section{Trivia}{


A class that only implements \code{fmap()} is called a functor.
}

\examples{
# We demonstrate by implementing a simple Either monad.
library(S7)

# Start by defining constructors of the Left and Right variants. Conventionally
# a Right variant signifies success and Left an error condition with a context.
left <- function(x) structure(list(value = x), class = c("left", "either"))
right <- function(x) structure(list(value = x), class = c("right", "either"))

# Implement fmap() and bind() methods to gain access to monad operators.
class_either <- new_S3_class("either")

method(fmap, class_either) <- function(m, f, ...) {
  if (inherits(m, "left")) m else right(f(m$value))
}

method(bind, class_either) <- function(m, f, ...) {
  if (inherits(m, "left")) m else f(m$value)
}

# Use with your function that handles errors by returning a monadic value.
mlog <- function(x) {
  if (x > 0) right(log(x)) else left("`x` must be strictly positive.")
}

# fmap() modifies the contained value with a regular function.
mlog(2) \%>>\% \(x) x + 1
mlog(0) \%>>\% \(x) x + 1

# bind() modifies the contained value with a function that returns an Either.
mlog(2) \%>-\% mlog()
mlog(0) \%>-\% mlog()
}
\seealso{
The \link[=monad-laws]{monad laws} and \link[=functor-laws]{functor
laws} that implementations should satisfy.

\link{List} and \link{Maybe} for examples of implementing classes.
}
