% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mrct_functions.R
\name{mrct}
\alias{mrct}
\title{Minimum regularized covariance trace estimator}
\usage{
mrct(
  data,
  h = 0.75,
  alpha = 0.01,
  initializations = 5,
  subset.iteration = 10,
  seed = 123,
  scaling.iterations = 10,
  scaling.tolerance = 10^(-4),
  criterion = "sum",
  sum.percentage = 0.75
)
}
\arguments{
\item{data}{Numeric matrix of a functional data set for which the esimator has to be calculated. Each row contains an observation. They are assumed to be observed on the same regular grid.}

\item{h}{Numeric value between \eqn{0.5} and \eqn{1}. Ratio of the data which the estimator is based on. Default is set to \eqn{0.75}, i.e. \eqn{75\%} of the data will be used for the estimator.}

\item{alpha}{Numeric (default is \eqn{0.01}). Tikhonov regularization parameter \eqn{\alpha}.}

\item{initializations}{Integer (default is \eqn{5}). Number of random initial subsets.}

\item{subset.iteration}{Integer (default is \eqn{10}). Maximum number of how often each subset is re-estimated and adjusted.}

\item{seed}{Integer (default is \eqn{123}). Random seed for reproducibility.}

\item{scaling.iterations}{Integer (default is \eqn{5}). The maximum number of times \eqn{k_1} is re-scaled if the error between subsequent
scalingparameters does not fall below \code{scaling.tolerance}.}

\item{scaling.tolerance}{Numeric (default is \eqn{10^{-4}}). The error tolerance for re-scaling. If the error falls below this value, the re-scaling procedure stops.}

\item{criterion}{Character. Criterion based on which the optimal subset is chosen among the final subsets. Possible options are: "\code{cluster}" and the default "\code{sum}".}

\item{sum.percentage}{Numeric value between \eqn{0.5} and \eqn{1}. Corresponding to the "\code{sum}" criterion. Determines the fraction of observations up to which the sum over the sorted functional Mahalanobis distances is calculated (in ascending order). Default is set to \eqn{0.75}, i.e. the sum of the smallest \eqn{75\%} of Mahalanobis distances is calculated. If outliers are present, this value should not be to high, in order not to include any outlying curves.}
}
\value{
A list:
\item{theoretical}{Integer vector of the indices corresponding to the outliers based on the MRCT estimator.}
\item{theoretical.w}{Same as \code{theoretical} with an additional re-weighting step.}
\item{aMHD}{Numeric vector containing the functional Mahalanobis distances of all observations based on the MRCT estimator.}
\item{aMHD.w}{Same as \code{aMHD} with an additional re-weighting step.}
\item{quant}{Numeric. Theoretical cutoff value for outlier detection.}
\item{quant.w}{Same as \code{quant} with an additional re-weighting step.}
\item{k}{Numeric. Scalingparameter \eqn{k_1} of Algorithm 1 described in \insertCite{oguamalam2023minimum}{mrct}.}
\item{k.w}{Same as \code{k} with an additional re-weighting step.}
\item{optimal.subset}{Integer vector of the optimal h-subset.}
\item{subsets}{Numeric matrix containing all final subsets. Each row of \code{subsets} is one final subset.}
\item{objval}{Numeric vector with the objective values of the final subsets based on \code{criterion}.}
}
\description{
Functional outlier detection based on the \code{minimum} \code{regularized} \code{covariance} \code{trace} estimator \insertCite{oguamalam2023minimum}{mrct} as a robust covariance estimator.
This estimator uses a generalization of the Mahalanobis distance for the functional setting \insertCite{berrendero2020}{mrct} and a corresponding theoretical cutoff value.
}
\examples{
# Fix seed for reproducibility
set.seed(123)

# Sample outlying indices
cont.ind <- sample(1:50, size=10)

# Generate 50 curves on the interval [0,1] at 50 timepoints with 20\% outliers
y <- mrct.rgauss(x.grid=seq(0,1,length.out=50), N=50, model=1,
                 outliers=cont.ind, method="linear")

# Visualize curves (regular curves grey, outliers black)
colormap <- rep("grey",50); colormap[cont.ind] <- "black"
matplot(x=seq(0,1,length.out=50), y=t(y), type="l", lty="solid",
        col=colormap, xlab="t",ylab="")

# Run MRCT
mrct.y <- mrct(data=y, h=0.75, alpha=0.1,
               initializations=10, criterion="sum")

# Visualize alpha-Mahalanobis distance with cutoff (horizontal black line)
# Colors correspond to simulated outliers, shapes to estimated (MRCT) ones
# (circle regular and triangle irregular curves)
shapemap <- rep(1,50); shapemap[mrct.y$theoretical.w] <- 2
plot(x=1:50, y=mrct.y$aMHD.w, col=colormap, pch=shapemap,
     xlab="Index", ylab=expression(alpha*"-MHD"))
abline(h = mrct.y$quant.w)

# If you dont have any information on possible outliers,
# alternatively you could use the S3 method plot.mrct()
mrct.plot(mrct.y)
}
\references{
\insertRef{berrendero2020}{mrct}.

\insertRef{oguamalam2023minimum}{mrct}.
}
