% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mxsem.R
\encoding{UTF-8}
\name{mxsem}
\alias{mxsem}
\title{mxsem}
\usage{
mxsem(
  model,
  data,
  scale_loadings = TRUE,
  scale_latent_variances = FALSE,
  add_intercepts = TRUE,
  add_variances = TRUE,
  add_exogenous_latent_covariances = TRUE,
  add_exogenous_manifest_covariances = TRUE,
  lbound_variances = TRUE,
  directed = unicode_directed(),
  undirected = unicode_undirected(),
  return_parameter_table = FALSE
)
}
\arguments{
\item{model}{model syntax similar to \strong{lavaan}'s syntax}

\item{data}{raw data used to fit the model. Alternatively, an object created
with \code{OpenMx::mxData} can be used (e.g., \code{OpenMx::mxData(observed = cov(OpenMx::Bollen), means = colMeans(OpenMx::Bollen), numObs = nrow(OpenMx::Bollen), type = "cov")}).}

\item{scale_loadings}{should the first loading of each latent variable be used for scaling?}

\item{scale_latent_variances}{should the latent variances be used for scaling?}

\item{add_intercepts}{should intercepts for manifest variables be added automatically? If set to false, intercepts must be added manually. If no intercepts
are added, \strong{mxsem} will automatically use just the observed covariances and not the observed means.}

\item{add_variances}{should variances for manifest and latent variables be added automatically?}

\item{add_exogenous_latent_covariances}{should covariances between exogenous latent variables be
added automatically?}

\item{add_exogenous_manifest_covariances}{should covariances between exogenous manifest variables be
added automatically?}

\item{lbound_variances}{should the lower bound for variances be set to 0.000001?}

\item{directed}{symbol used to indicate directed effects (regressions and loadings)}

\item{undirected}{symbol used to indicate undirected effects (variances and covariances)}

\item{return_parameter_table}{if set to TRUE, the internal parameter table is returend
together with the mxModel}
}
\value{
mxModel object that can be fitted with mxRun or mxTryHard. If return_parameter_table
is TRUE, a list with the mxModel and the parameter table is returned.
}
\description{
Create an extended SEM with \strong{OpenMx} (Boker et al., 2011) using a
\strong{lavaan}-style (Rosseel, 2012) syntax.
}
\details{
Setting up SEM can be tedious. The \strong{lavaan} (Rosseel, 2012) package provides a great syntax to
make the process easier. The objective of \strong{mxsem} is to provide a similar syntax
for \strong{OpenMx}. \strong{OpenMx} is a flexible R package for extended SEM. However, note that
\strong{mxsem} only covers a small part of the \strong{OpenMx} framework by focusing on "standard"
SEM. Similar to \strong{lavaan}'s \code{sem()}-function, \code{mxsem} tries to set up parts
of the model automatically (e.g., adding variances automatically or scaling the
latent variables automatically). If you want to unlock
the full potential of \strong{OpenMx}, \strong{mxsem} may not be the best option.

\strong{Warning}: The syntax and settings of \strong{mxsem} may differ from
\strong{lavaan} in some cases. See \code{vignette("Syntax", package = "mxsem")} for more details
on the syntax and the default arguments.
\subsection{Alternatives}{

You will find similar functions in the following packages:
\itemize{
\item \href{https://github.com/mikewlcheung/metasem}{\strong{metaSEM}} (Cheung, 2015) provides a \code{lavaan2RAM}
function that can be combined with the \code{create.mxModel} function. This combination
offers more features than \strong{mxsem}. For instance, constraints of the form \code{a < b}
are supported. In \strong{mxsem} such constraints require algebras (e.g., \verb{!diff; a := b - exp(diff)}).
\item \href{https://github.com/tbates/umx}{\strong{umx}} (Bates et al., 2019)
provides the \code{umxRAM} and \code{umxLav2RAM} functions that can parse single \strong{lavaan}-style
statements (e.g., \code{eta =~ y1 + y2 + y3})
or an entire \strong{lavaan} models to \strong{OpenMx} models.
\item \href{https://github.com/cjvanlissa/tidySEM}{\strong{tidySEM}} (van Lissa, 2023) provides the
\code{as_ram} function to translate \strong{lavaan} syntax to \strong{OpenMx} and also implements a unified syntax to
specify both, \strong{lavaan} and \strong{OpenMx} models. Additionally, it works well with the
\strong{tidyverse}.
\item \href{https://github.com/OpenMx/ezMx}{\strong{ezMx}} (Bates, et al. 2014) simplifies fitting SEM with \strong{OpenMx}
and also provides a translation of \strong{lavaan} models to \strong{OpenMx} with the
\code{lavaan.to.OpenMx} function.
}

Because \strong{mxsem} implements the syntax parser from scratch, it can extend the
\strong{lavaan} syntax to account for specific \strong{OpenMx} features. This enables
implicit transformations with curly braces.
}

\subsection{Citation}{

Cite \strong{OpenMx} (Boker et al., 2011) for the modeling and \strong{lavaan} for the
syntax (Rosseel, 2012). \strong{mxsem} itself is just a very small package and lets
\strong{OpenMx} do all the heavy lifting.
}

\subsection{Defaults}{

By default, \strong{mxsem} scales latent variables by setting the loadings on the first
item to 1. This can be changed by setting \code{scale_loadings = FALSE} in the function
call. Setting \code{scale_latent_variances = TRUE} sets latent variances to 1 for
scaling.

\strong{mxsem} will add intercepts for all manifest variables as well as variances for
all manifest and latent variables. A lower bound of 1e-6 will be added to all
variances. Finally, covariances for all exogenous variables will be added.
All of these options can be changed when calling \strong{mxsem}.
}

\subsection{Syntax}{

The syntax is, for the most part, identical to that of \strong{lavaan}. The following
specifies loadings of a latent variable \code{eta} on manifest variables \code{y1}-\code{y4}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{eta =~ y1 + y2 + y3
}\if{html}{\out{</div>}}

Regressions are specified with \code{~}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{xi  =~ x1 + x2 + x3
eta =~ y1 + y2 + y3
# predict eta with xi:
eta ~  xi
}\if{html}{\out{</div>}}

Add covariances with \verb{~~}

\if{html}{\out{<div class="sourceCode">}}\preformatted{xi  =~ x1 + x2 + x3
eta =~ y1 + y2 + y3
# predict eta with xi:
eta ~  xi
x1 ~~ x2
}\if{html}{\out{</div>}}

Intercepts are specified with \code{~1}

\if{html}{\out{<div class="sourceCode">}}\preformatted{xi  =~ x1 + x2 + x3
eta =~ y1 + y2 + y3
# predict eta with xi:
eta ~  xi
x1 ~~ x2

eta ~ 1
}\if{html}{\out{</div>}}
}

\subsection{Parameter labels and constraints}{

Add labels to parameters as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{xi  =~ l1*x1 + l2*x2 + l3*x3
eta =~ l4*y1 + l5*y2 + l6*y3
# predict eta with xi:
eta ~  b*xi
}\if{html}{\out{</div>}}

Fix parameters by using numeric values instead of labels:

\if{html}{\out{<div class="sourceCode">}}\preformatted{xi  =~ 1*x1 + l2*x2 + l3*x3
eta =~ 1*y1 + l5*y2 + l6*y3
# predict eta with xi:
eta ~  b*xi
}\if{html}{\out{</div>}}
}

\subsection{Bounds}{

Lower and upper bounds allow for constraints on parameters. For instance,
a lower bound can prevent negative variances.

\if{html}{\out{<div class="sourceCode">}}\preformatted{xi  =~ 1*x1 + l2*x2 + l3*x3
eta =~ 1*y1 + l5*y2 + l6*y3
# predict eta with xi:
eta ~  b*xi
# residual variance for x1
x1 ~~ v*x1
# bound:
v > 0
}\if{html}{\out{</div>}}

Upper bounds are specified with v < 10. Note that the parameter label must always
come first. The following is not allowed: \code{0 < v} or \code{10 > v}.
}

\subsection{(Non-)linear constraints}{

Assume that latent construct \code{eta} was observed twice, where \code{eta1} is the first
observation and \code{eta2} the second. We want to define the loadings of \code{eta2}
on its observations as \code{l_1 + delta_l1}. If \code{delta_l1} is zero, we have measurement
invariance.

\if{html}{\out{<div class="sourceCode">}}\preformatted{eta1 =~ l1*y1 + l2*y2 + l3*y3
eta2 =~ l4*y4 + l5*y5 + l6*y6
# define new delta-parameter
!delta_1; !delta_2; !delta_3
# redefine l4-l6
l4 := l1 + delta_1
l5 := l2 + delta_2
l6 := l3 + delta_3
}\if{html}{\out{</div>}}

Alternatively, implicit transformations can be used as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{eta1 =~ l1*y1 + l2*y2 + l3*y3
eta2 =~ \{l1 + delta_1\} * y4 + \{l2 + delta_2\} * y5 + \{l3 + delta_3\} * y6
}\if{html}{\out{</div>}}

Specific labels for the transformation results can also be provided:

\if{html}{\out{<div class="sourceCode">}}\preformatted{eta1 =~ l1*y1 + l2*y2 + l3*y3
eta2 =~ \{l4 := l1 + delta_1\} * y4 + \{l5 := l2 + delta_2\} * y5 + \{l6 := l3 + delta_3\} * y6
}\if{html}{\out{</div>}}

This is inspired by the approach in \strong{metaSEM} (Cheung, 2015).
}

\subsection{Definition variables}{

Definition variables allow for person-specific parameter constraints. Use the
\code{data.}-prefix to specify definition variables.

\if{html}{\out{<div class="sourceCode">}}\preformatted{I =~ 1*y1 + 1*y2 + 1*y3 + 1*y4 + 1*y5
S =~ data.t_1 * y1 + data.t_2 * y2 + data.t_3 * y3 + data.t_4 * y4 + data.t_5 * y5

I ~ int*1
S ~ slp*1
}\if{html}{\out{</div>}}
}

\subsection{Starting Values}{

\strong{mxsem} differs from \strong{lavaan} in the specification of starting values. Instead
of providing starting values in the model syntax, the \code{set_starting_values}
function is used.
}

\subsection{References}{
\itemize{
\item Bates, T. C., Maes, H., & Neale, M. C. (2019). umx: Twin and Path-Based Structural Equation Modeling in R. Twin Research and Human Genetics, 22(1), 27–41. https://doi.org/10.1017/thg.2019.2
\item Bates, T. C., Prindle, J. J. (2014). ezMx. https://github.com/OpenMx/ezMx
\item Boker, S. M., Neale, M., Maes, H., Wilde, M., Spiegel, M., Brick, T., Spies, J., Estabrook, R., Kenny, S., Bates, T., Mehta, P., & Fox, J. (2011).
OpenMx: An Open Source Extended Structural Equation Modeling Framework. Psychometrika, 76(2), 306–317. https://doi.org/10.1007/s11336-010-9200-6
\item Cheung, M. W.-L. (2015). metaSEM: An R package for meta-analysis using structural equation modeling. Frontiers in Psychology, 5. https://doi.org/10.3389/fpsyg.2014.01521
\item Rosseel, Y. (2012). lavaan: An R package for structural equation modeling. Journal of Statistical Software, 48(2), 1–36. https://doi.org/10.18637/jss.v048.i02
\item van Lissa, C. J. (2023). tidySEM: Tidy Structural Equation Modeling. R package version 0.2.4, https://cjvanlissa.github.io/tidySEM/.
}
}
}
\examples{
# THE FOLLOWING EXAMPLE IS ADAPTED FROM LAVAAN
library(mxsem)

model <- '
  # latent variable definitions
     ind60 =~ x1 + x2 + x3
     dem60 =~ y1 + a1*y2 + b*y3 + c1*y4
     dem65 =~ y5 + a2*y6 + b*y7 + c2*y8

  # regressions
    dem60 ~ ind60
    dem65 ~ ind60 + dem60

  # residual correlations
    y1 ~~ y5
    y2 ~~ y4 + y6
    y3 ~~ y7
    y4 ~~ y8
    y6 ~~ y8
'

fit <- mxsem(model = model,
            data  = OpenMx::Bollen) |>
  mxTryHard()
omxGetParameters(fit)


model_transformations <- '
  # latent variable definitions
     ind60 =~ x1 + x2 + x3
     dem60 =~ y1 + a1*y2 + b1*y3 + c1*y4
     dem65 =~ y5 + {a2 := a1 + delta_a}*y6 + {b2 := b1 + delta_b}*y7 + c2*y8

  # regressions
    dem60 ~ ind60
    dem65 ~ ind60 + dem60

  # residual correlations
    y1 ~~ y5
    y2 ~~ y4 + y6
    y3 ~~ y7
    y4 ~~ y8
    y6 ~~ y8
'

fit <- mxsem(model = model_transformations,
            data  = OpenMx::Bollen) |>
  mxTryHard()
omxGetParameters(fit)
}
