% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quadratureGrids.R
\name{configureQuadGrid}
\alias{configureQuadGrid}
\title{Configure Quadrature Grids}
\usage{
configureQuadGrid(d = 1, levels = 3, quadRule = "AGHQ", control = list())
}
\arguments{
\item{d}{Number of dimensions.}

\item{levels}{Number of quadrature points to generate in each dimension (or the level of accuracy of a sparse grid).}

\item{quadRule}{Default quadRule to be used. Options are "AGHQ" or "CCD". May also be a user supplied quadrature rule as a nimbleFunction.}

\item{control}{list to control how the quadrature rules are built. See details for options.}
}
\description{
Takes requested quadrature rules, builds the associated quadrature grids, and caches them. Updates 
all the features of the grids on call and returns nodes and weights.
}
\details{
Different options for building quadrature rules can be specified by `control` list.  These include

\itemize{
   \item \code{quadRules} which includes all the different rules that are being requested.
   \item \code{constructionRule} How each quadrature rule should be combined into multiple dimension. Currently possible to choose "product" which repeats
     creates a grid of repeated nodes in each dimension. Alternatively, can use "sparse" to apply standard sparse construction.
   \item \code{CCD_f0} multiplier for the CCD grid for how much past the radius sqrt(d) to extend the nodes. Unless an advanced user, keep at default of 1.1.
   \item \code{prune} the proportion of quadrature points (when generated by the product rule) to keep based on the weights for integration over a multivariate normal.
   \item \code{userConstruction} choose method to construct multivariate grid. If "MULTI" then user provided a multivariate construction in the provided function. If "PRODUCT"
   then a product rule construction is used to generate quadrature points in each dimension. If "SPARSE", then a Smolyak rule is applied.
}

Quadrature grids are generally based on adaptive Gauss-Hermite (GH) quadrature which is expanded via a product or sparse rule into multiple dimensions. Sparse grids are
built following the Smolyak rule (Heiss and Winschel, 2008) and demonstrated in the package \pkg{mvQuad} (Weiser, 2023). Pruning is also implemented as described in 
\enc{Jäckel}{Jaeckel} (2005), where weights are adjusted by the value of a standard multivariate normal at that node, and nodes are removed until some threshold is met.

The available methods that can called by this function once it is setup are:

\itemize{
  \item \code{buildGrid} method for creating the quadrature grid. Inputs are \code{method} includes ("AGHQ", "CCD", "USER") to choose the active quadrature rule. 
  \code{nQuad} number of quadrature points (nQuad) per dimension, \code{prune} proportion of points in the product construction to use, and \code{constructionRule} 
  includes ("product", "sparse"). Default behaviour for all input is to use values that were last requested.

  \item \code{setDim} Allows the user to change the dimension of the quadrature grids which will reset all grids in use.

  \item \code{nodes}, \code{weights}, \code{gridSize}, and \code{modeIndex} give access to the user for the details of the quadrature grid in use. This is either based
  on the last call to \code{buildGrid}, or by choosing a different grid with \code{setMethod}. \code{nodes} and \code{weights} return all nodes and weights if no values
  are passed, or if an index is passed, the node and weight associated with that index. Passing -1 indicates that the mode should be returned which in this case is all zeros.
}
}
\examples{

library(mvQuad)
RmvQuad <- function(levels, d) {
   out <- mvQuad::createNIGrid(dim=d, type = "GHe", level=levels, ndConstruction = "sparse")
   cbind(out$weights, out$nodes)
}
nimMVQuad <- nimbleRcall(function(levels = integer(), d = integer()){},
                         Rfun = "RmvQuad", returnType = double(2))
myQuadRule <- nimbleFunction(
     contains = QUAD_RULE_BASE,
     name = "quadRule_USER",
     setup = function() {},
     run = function() {},
     methods = list(
         buildGrid = function(levels = integer(0, default = 0), d = integer(0, default = 1)) {
             output <- nimMVQuad(levels, d)
             returnType(double(2))
             return(output)
         }
     )
 )

quadGrid_user <- configureQuadGrid(d=2, levels=3, quadRule = myQuadRule,
                   control = list(quadRules = c("AGHQ", "CCD", "AGHQSPARSE"),
                             userConstruction = "MULTI"))

}
\references{
Heiss, F. and Winschel V. (2008). Likelihood approximation by numerical integration on sparse grids. Journal of Econometrics 144 (1), 62–80.

Weiser, C. (2023). _mvQuad: Methods for Multivariate Quadrature._. (R package version 1.0-8), <https://CRAN.R-project.org/package=mvQuad>.

\enc{Jäckel}{Jaeckel}, P. (2005). A note on multivariate gauss-hermite quadrature. London: ABN-Amro. Re.
}
\author{
Paul van Dam-Bates
}
