np.lm.test <-
  function(formula, data, ..., anova.test = TRUE,
           method = "perm", homosced = FALSE, lambda = 0, 
           R = 9999, parallel = FALSE, cl = NULL,
           perm.dist = TRUE, na.rm = TRUE){
    # Nonparametric Tests of Linear Model Coefficients/Terms
    # Nathaniel E. Helwig (helwig@umn.edu)
    # last updated: 2026-01-15
    
    
    # Note: below code adapted from R's lm() function (to parse formula input)
    mf <- match.call(expand.dots = TRUE)
    m <- match(c("formula", "data", "subset", "weights", "na.action", 
                 "offset"), names(mf), 0L)
    mf <- mf[c(1L, m)]
    mf$drop.unused.levels <- TRUE
    mf[[1L]] <- quote(stats::model.frame)
    mf <- eval(mf, parent.frame())
    mt <- attr(mf, "terms")
    
    # get y, w, and offset
    y <- model.response(mf, "numeric")
    w <- as.vector(model.weights(mf))
    if (!is.null(w) && !is.numeric(w)) 
      stop("'weights' must be a numeric vector")
    offset <- model.offset(mf)
    
    # basic checks
    mlm <- is.matrix(y)
    if(mlm && ncol(y) == 1L){
      mlm <- FALSE
      y <- as.numeric(y)
    }
    ny <- if (mlm) nrow(y) else length(y)
    if(mlm) stop("Multivariate use of np.lm.test() is not supported.\nUse the np.reg.test() function for multivariate linear model tests.")
    if(is.null(offset)){
      offset <- rep(0.0, ny)
    } else {
      if (!mlm) 
        offset <- as.vector(offset)
      if (NROW(offset) != ny) 
        stop(gettextf("number of offsets is %d, should equal %d (number of observations)", 
                      NROW(offset), ny), domain = NA)
    }
    
    # build design matrix
    args <- list(...)
    contrasts <- args$contrasts
    x <- model.matrix(mt, mf, contrasts)
    
    # get term labels and term to column assignments
    term.labs <- attr(mt, "term.labels")
    assign <- attr(x, "assign")
    if(colnames(x)[1] == "(Intercept)"){
      assign <- assign[-1]
      x <- x[,-1]
    } 
    
    # na.omit
    if(na.rm){
      x <- na.omit(x)
      y <- na.omit(y)
      w <- na.omit(w)
      offset <- na.omit(offset)
    }
    
    # implement test
    pt <- rand.test.lm0(x = x, y = y, weights = w, offset = offset, 
                        anova = anova.test, assign = assign, 
                        term.labels = term.labs, method = method, 
                        homosced = homosced, lambda = lambda, 
                        R = R, parallel = parallel, cl = cl,
                        perm.dist = perm.dist, na.rm = na.rm)
    pt$anova.test <- anova.test
    pt$formula <- formula
    pt$data <- data
    
    ### return results
    class(pt) <- "np.lm.test"
    return(pt)
    
  } # end np.lm.test
