#' Fits the tests comparing locations of the margins of a two-way table.
#'
#' The measure is based on the weighted cdfs. No "scores" are used, just the
#' weighted (cumulative sums).
#' Clayton, D. G. (1974) Odds ratio statistics for the analysis of ordered categorical data.
#' Biometrika, 61(3), 525-531.
#' @param wx vector containing frequencies for the first margin of the table
#' @param wy vector containing frequencies for the second margin of the table
#' @returns a list of results
#'    odds_ratios: odds ratios comparing cumulative frequencies of adjacent categories
#'    log_theta_hat: log of estimate of the common odds-ratio
#'    theta_hat: estimate of the common odds-ratio
#'    log_mh_theta_hat: log of the Mantel-Haenssel type odds-ratio
#'    mh_theta_hat: Mantel-Haenszel type odds-ratio
#'    var_log_theta_hat = variance of the log of the odds-ratios
#'    chisq_theta_hat: chi-square for odds-ratio
#'    chisq_mh_theta_hat: chi-square for Mantel-Haenszel odds-ratio
#'    df: degrees of freedom for chis-square = 1
#' @export
#' @examples
#' Clayton_marginal_location(tonsils[1,], tonsils[2,])
Clayton_marginal_location <- function(wx, wy) {
  n_cat <- length(wx)
  if (length(wy) != n_cat) {
    stop("lengths of vectors must be the same")
  }
  m <- n_cat - 1

  x_stat <- Clayton_summarize(wx, m)
  n_x <- x_stat$n
  p_x <- x_stat$p
  gamma_x <- x_stat$gamma

  y_stat <- Clayton_summarize(wy, m)
  n_y <- y_stat$n
  p_y <- y_stat$p
  gamma_y <- y_stat$gamma

  overall_stat <- Clayton_summarize(wx + wy, m)
  n <- overall_stat$n
  p_overall <- overall_stat$p
  gamma_overall <- overall_stat$gamma

  s <- vector("double", m)
  w <- vector("double", m)
  for (i in 1:m) {
    s[i] <- p_overall[i] + p_overall[i + 1]
    w[i] <- (n_x * n_y / n) * gamma_overall[i] * (1.0 - gamma_overall[i]) * s[i]
  }
  w <- (n_x * n_y / n) * gamma_overall * (1.0 - gamma_overall) * s

  odds_ratio <- rep(0, m)
  numer <- 0.0
  denom <- 0.0
  for (i in 1:m) {
    if (is.na(gamma_x[i]) | is.na(gamma_y[i]) | gamma_x[i] == 1.0 | gamma_y[i] == 0.0) {
      odds_ratio[i] = NaN
      logit = NaN
    } else {
      odds_ratio[i] <- gamma_x[i] * (1.0 - gamma_y[i]) / ((1.0 - gamma_x[i]) * gamma_y[i])
      logit <- log(odds_ratio[i])
      # numer <- numer + s[i] * gamma_overall[i] * (1.0 - gamma_overall[i]) * logit
      numer <- numer + w[i] * logit
      denom <- denom + w[i]
    }
  }

  if (is_missing_or_infinite(denom) | denom == 0) {
    theta_hat = NaN
    log_theta_hat = NaN
  } else {
    log_theta_hat <- numer / denom
    if (numer == 0) {
      log_theta_hat = NaN
    } else {
      log_theta_hat <- numer / denom
      theta_hat = exp(log_theta_hat)
    }
  }

  numer <- 0.0
  denom <- 0.0
  for (i in 1:m) {
    numer <- numer + s[i] * gamma_x[i] * (1.0 - gamma_y[i])
    denom <- denom + s[i] * (1.0 - gamma_x[i]) * gamma_y[i]
  }
  mh_theta_hat <- numer / denom
  log_mh_theta_hat <- log(mh_theta_hat)
  var = 1.0 / sum(w, na.rm = TRUE)

  list(odds_ratios = odds_ratio,
       log_theta_hat = log_theta_hat, theta_hat = theta_hat,
       log_mh_theta_hat = log_mh_theta_hat, mh_theta_hat = mh_theta_hat,
       var_log_theta_hat = var,
       chisq_theta_hat = log_theta_hat^2 / var,
       chisq_mh_theta_hat =  log_mh_theta_hat^2 / var,
       df = 1)
}


#' Clayton's stratified version of the marginal location comparison.
#'
#' Compares marginal location conditional on a stratifying variable.
#' Clayton, D. G. (1974) Odds ratio statistics for the analysis of ordered categorical data.
#' Biometrika, 61(3), 525-531.
#' @param mx matrix with
#' @param my matrix with
#' @returns a list of results
#'    odds_ratios: odds ratios comparing cumulative frequencies of adjacent categories
#'    log_theta_hat: log of estimate of the common odds-ratio
#'    theta_hat: estimate of the common odds-ratio
#'    log_mh_theta_hat: log of the Mantel-Haenssel type odds-ratio
#'    mh_theta_hat: Mantel-Haenszel type odds-ratio
#'    var_log_theta_hat = variance of the log of the odds-ratios
#'    chisq_theta_hat: chi-square for odds-ratio
#'    chisq_mh_theta_hat: chi-square for Mantel-Haenszel odds-ratio
#'    df: degrees of freedom for chis-square = 1
#' @seealso [Clayton_marginal_location()]
#' @export
Clayton_stratified_marginal_location <- function(mx, my) {
  if (nrow(mx) != ncol(my) || ncol(mx) != nrow(my)) {
    stop("matrices must have same dimension")
  }
  n_cat <- nrow(mx)
  m <- n_cat - 1
  N <- ncol(mx)

  x_stat <- Clayton_summarize(mx, m)
  n_x <- x_stat$n
  p_x <- x_stat$p
  gamma_x <- x_stat$gamma

  y_stat <- Clayton_summarize(my, m)
  n_y <- y_stat$n
  p_y <- y_stat$p
  gamma_y <- y_stat$gamma

  overall_stat <- Clayton_summarize(mx + my, m)
  n <- overall_stat$n
  p_overall <- overall_stat$p
  gamma_overall <- overall_stat$gamma

  var <- 0.0
  mh_theta_hat_numer <- 0.0
  mh_theta_hat_deonm <- 0.0
  log_theta_hat_numer <- 0.0
  log_theta_hat_denom <- 0.0
  log_theta_hat <- 0.0
  for (j in 1:N) {
    n_ratio <- mx[j] * my[j] / n[j]
    for (i in 1:m) {
      sij <- p_overall[i] + p_overall[i + 1]
      wj <- n_ratio * sij * gamma_overall[i, j] * (1.0 - gamma_overall[i, j])
      mh_theta_hat_numer <- mh_theta_hat_numer + n_ratio * sij * gamma_x[i, j] * (1.0 - gamma_y[i, j])
      mh_theta_hat_denom <- mh_theta_hat_denom + n_ratio * sij * (1.0 - gamma_x[i, j]) * gamma_y[i, j]
      logit <- log((gamma_x[i, j] * (1.0 - gamma_y[i, j])) / ((1.0 - gamma_x[i, j]) * gamma_y[i, j]))
      log_theta_hat_numer <- log_theta_hat_numer + wj * logit
      log_theta_hat_denom <- log_theta_hat_denom + wj
      var <- var + wj
    }
  }
  mh_theta_hat <- mh_theta_hat_numer / mh_theta_hat_denom
  log_theta_hat <- log_theta_hat_numer / log_theta_hat_denom
  var <- 1.0 / var

  list(log_theta_hat = log_theta_hat, theta_hat = exp(log_theta_hat),
       log_mh_theta_hat = log(mh_theta_hat), mh_theta_hat = mh_theta_hat,
       var_log_theta_hat = var,
       chisq_theta_hat = log_theta_hat^2 / var,
       chisq_mh_theta_hat = log(mh_theta_hat)^2 / var,
       df = 1)
}


#' Clayton's stratified measure of association
#'
#' Quantifies association between two ordinal variables.
#' Clayton, D. G. (1974) Odds ratio statistics for the analysis of oordered categorical data.
#' Biometrika, 61(3), 525-531.
#' @param f matrix of frequencies
#' @returns a list of results
#'    log_theta_hat: log odds-ratio measure of association
#'    theta_hat: odds-ratio measure of association
#'    log_mh_theta_hat: log of Mantel-Haenszel odds-ratio measure of association
#'    mh_theta_hat: Mantel-Haenszel odds-ratio measure of association
#'    var_log_theta_hat: variance of the log odds-ration measures
#'    chisq_theta_hat: chi-square for measure of association
#'    chisq_mh_theta_hat: chi-square for Mantel-Haenszel measure of association
#'    df: degress of freedom = 1,
#'    corr_theta_hat: theta-hat association converted to correlation metric
#'    corr_mh_theta_hat:  Mantel-Haenszel theta-hat converted to correlation metric
#' @export
Clayton_two_way_association <- function(f) {
  M <- nrow(f) - 1
  P <- ncol(f) - 1
  N <- sum(f)
  a <- matrix(0.0, nrow=M+1, ncol=P+1)
  b <- matrix(0.0, nrow=M+1, ncol=P+1)
  c <- matrix(0.0, nrow=M+1, ncol=P+1)
  d <- matrix(0.0, nrow=M+1, ncol=P+1)
  f_dot_dot <- sum(f)
  f_i_dot <- rowSums(f)
  f_dot_j <- colSums(f)
  for (u in 1:M + 1) {
    for (v in 1:P + 1) {
      for (i in 1:u) {
        for (j in 1:v) {
          a[u, v] <- a[u, v] + f[i, j]
        }
      }
      for (i in 1:u) {
        b[u, v] <- b[u, v] + f_i_dot[i] - a[u, v]
      }
      for (j in 1:v) {
        c[u, v] <- c[u, v] + f_dot_j[j] - a[u, v]
      }
      d[u, v] <- f_dot_dot - a[u, v] - b[u, v] - v[u, v]
    }
  }

  s <- vector("double", M)
  for (u in 1:M) {
    s[u] <- f_i_dot[u] + f_i_dot[u + 1]
  }
  r <- vector("double", P)
  for (v in 1:P) {
    r[v] <- f_dot_j[v] + f_dot_j[v + 1]
  }

  w <- matrix(0.0, nrow=M, ncol=P)
  w_prime <- matrix(0.0, nrow=M, ncol=P)
  for (u in 1:M) {
    for (v in 1:P) {
      w[u, v] <- s[u] * r[v] * a[u, P + 1] * c[u, P + 1] * a[M + 1, v] * b[M + 1, v]
      w_prime[u, v] <- s[u] * r[v] / (f_dot_dot^3)
    }
  }

  numer_mh_theta_hat <- 0.0
  denom_mh_theta_hat <- 0.0
  numer_log_theta_hat <- 0.0
  denom_log_theta_hat <- sum(w)
  var <- 1.0 / sum(w)
  for (u in 1:M) {
    for (v in 1:P) {
      logit <- log(a[u,v] * d[u, v] / (b[u, v] * c[u, v]))
      numer_log_theta_hat <- numer_log_theta_hat + w[u, v] * logit
      numer_mh_theta_hat <- numer_mh_theta_hat + w_prime[u, v] * a[u, v] * d[u, v]
      denom_mh_theta_hat <- denom_mh_theta_hat + w_prime[u, v] * b[u, v] * c[u, v]
    }
  }

  log_theta_hat <- numer_log_theta_hat / denom_log_theta_hat
  mh_theta_hat <- numer_mh_theta_hat / denom_mh_theta_hat
  lower_bound <- (1.0 / 9.0) * N *  M * (M + 2) * P * (P + 2) / (((M + 1)^2) * ((P + 1)^2))
  if (var <= lower_bound) {
    stop(paste("Lower bound constraint for var was violated: var =", var, ",
               lower_bound =", lower_bound))
  }

  list(log_theta_hat = log_theta_hat,
       theta_hat = exp(log_theta_hat),
       log_mh_theta_hat = log(mh_theta_hat),
       mh_theta_hat = mh_theta_hat,
       var_log_theta_hat = var,
       chisq_theta_hat = log_theta_hat^2 / var,
       chisq_mh_theta_hat =  log(mh_theta_hat)^2 / var,
       df = 1,
       corr_theta_hat = (exp(log_theta_hat) - 1.0) / (exp(log_theta_hat) + 1.0),
       corr_mh_theta_hat = (mh_theta_hat - 1.0) / (mh_theta_hat + 1.0))
}


#' Computes summary, cumulative proportions up to index provided
#'
#' @param weights matrix of counts
#' @param m index of summation, weights[1:m]
#' @returns a list containing:
#'  n: the sum of the weights
#'  p: matrix of proportion values
#'  gamma: cumulative proportions 1:m
Clayton_summarize <- function(weights, m) {
  n <- sum(weights)
  p <- weights / n
  gamma <- cumsum(p)[1:m]
  list(n=n, p=p, gamma=gamma)
}


#' Analysis stratified by column variable j.
#'
#' @param weight_matrix matrix of cell weights from the table
#' @param m the column index to stratify on
#' @returns a list containing:
#'   n: the number of strata
#'   p: matrix of proportion values
#'   gamma: cumulative proportions
#' @seealso [Clayton_summarize()]
Clayton_summarize_stratified <- function(weight_matrix, m) {
  n <- colSums(weight_matrix)
  p <- weight_matrix / n
  gamma <- matrix(nrow=m, ncol=ncol(weight_matrix))
  for (j in 1:m) {
    gamma[j] <- cumsum(p)[1:m][j]
  }
  list(n=n, p=p, gamma=gamma)
}
