% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/openai_parse.R
\name{parse_openai_batch_output}
\alias{parse_openai_batch_output}
\title{Parse an OpenAI Batch output JSONL file}
\usage{
parse_openai_batch_output(
  path,
  tag_prefix = "<BETTER_SAMPLE>",
  tag_suffix = "</BETTER_SAMPLE>"
)
}
\arguments{
\item{path}{Path to a JSONL output file downloaded from the OpenAI Batch
API.}

\item{tag_prefix}{Character string marking the start of the better-sample
tag. Defaults to \code{"<BETTER_SAMPLE>"}.}

\item{tag_suffix}{Character string marking the end of the better-sample
tag. Defaults to \code{"</BETTER_SAMPLE>"}.}
}
\value{
A tibble with one row per successfully parsed comparison and
columns:
\describe{
\item{custom_id}{The \code{custom_id} from the batch request.}
\item{ID1, ID2}{Sample IDs inferred from \code{custom_id}.}
\item{model}{The model name reported by the API.}
\item{object_type}{The OpenAI response object type
(e.g., \code{"chat.completion"} or \code{"response"}).}
\item{status_code}{HTTP-style status code from the batch output.}
\item{error_message}{Error message, if present; otherwise \code{NA}.}
\item{thoughts}{Reasoning / thinking summary text when available
(for Responses with reasoning); otherwise \code{NA}.}
\item{content}{The raw assistant visible content string (the LLM's
output), used to locate the \code{<BETTER_SAMPLE>} tag. For
Responses with reasoning this does not include reasoning
summaries, which are kept in \code{thoughts}.}
\item{better_sample}{Either \code{"SAMPLE_1"}, \code{"SAMPLE_2"},
or \code{NA} if the tag was not found.}
\item{better_id}{\code{ID1} if \code{SAMPLE_1} was chosen, \code{ID2}
if \code{SAMPLE_2} was chosen, or \code{NA}.}
\item{prompt_tokens}{Prompt/input token count (if reported).}
\item{completion_tokens}{Completion/output token count (if reported).}
\item{total_tokens}{Total tokens (if reported).}
\item{prompt_cached_tokens}{Cached prompt tokens (if reported via
\code{input_tokens_details$cached_tokens}); otherwise \code{NA}.}
\item{reasoning_tokens}{Reasoning tokens (if reported via
\code{output_tokens_details$reasoning_tokens}); otherwise
\code{NA}.}
}
}
\description{
This function reads an OpenAI Batch API output file (JSONL) and extracts
pairwise comparison results for use with Bradley–Terry models. It supports
both the Chat Completions endpoint (where \code{object = "chat.completion"})
and the Responses endpoint (where \code{object = "response"}), including
GPT-5.1 with reasoning.
}
\details{
For each line, the function:
\itemize{
\item extracts \code{custom_id} and parses \code{ID1} and \code{ID2}
from the pattern \code{"<prefix>ID1_vs_ID2"},
\item pulls the raw LLM content containing the
\code{<BETTER_SAMPLE>...</BETTER_SAMPLE>} tag,
\item determines whether \code{SAMPLE_1} or \code{SAMPLE_2} was
selected and maps that to \code{better_id},
\item collects model name and token usage statistics (including
reasoning tokens for GPT-5.1 Responses),
\item when using the Responses endpoint with reasoning, separates
reasoning summaries into the \code{thoughts} column and visible
assistant output into \code{content}.
}

The returned data frame is suitable as input for
\code{\link{build_bt_data}}.
}
\examples{
# Create a temporary JSONL file containing a simulated OpenAI batch result
tf <- tempfile(fileext = ".jsonl")

# A single line of JSON representing a successful Chat Completion
# custom_id implies "LIVE_" prefix, ID1="A", ID2="B"
json_line <- paste0(
  '{"custom_id": "LIVE_A_vs_B", ',
  '"response": {"status_code": 200, "body": {',
  '"object": "chat.completion", ',
  '"model": "gpt-4", ',
  '"choices": [{"message": {"content": "<BETTER_SAMPLE>SAMPLE_1</BETTER_SAMPLE>"}}], ',
  '"usage": {"prompt_tokens": 50, "completion_tokens": 10, "total_tokens": 60}}}}'
)

writeLines(json_line, tf)

# Parse the output
res <- parse_openai_batch_output(tf)

# Inspect the result
print(res$better_id)
print(res$prompt_tokens)

# Clean up
unlink(tf)

}
