#' @title
#' Lifting posets.
#'
#' @description
#' Lifts the input poset, i.e. adds a (possibly new) bottom element to it.
#'
#' @param poset An object of S4 class `POSet`.
#' Argument `poset1` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...).
#'
#' @param element A character string (the name of the added bottom).
#'
#' @return
#' The lifted poset, an object of S4 class `POSet`.
#'
#' @examples
#' elems <- c("a", "b", "c", "d")
#'
#' doms <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = doms)
#'
#' #Lifting
#' lifted.pos <- LiftingPOSet(pos, "bot")
#'
#' @name LiftingPOSet
#' @export LiftingPOSet
LiftingPOSet <- function(poset, element) {
  if (!methods::is(poset, "POSet")) {
    stop("Argument poset must be of class POSet")
  }
  if (!is.character(element) || length(element) != 1) {
    stop("element must be a string")
  }

  tryCatch({
    ptr <- .Call("_BuildLiftingPOSet", poset@ptr, element)
    result <- methods::new("POSet", ptr = ptr)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
