% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/psymet_functions.R
\name{cronbach}
\alias{cronbach}
\title{Cronbach's Alpha of a Set of Variables/Items}
\usage{
cronbach(
  data,
  vrb.nm,
  ci.type = "delta",
  level = 0.95,
  use = "pairwise.complete.obs",
  stats = c("average_r", "nvrb"),
  R = 200L,
  boot.ci.type = "perc"
)
}
\arguments{
\item{data}{data.frame of data.}

\item{vrb.nm}{character vector of colnames of \code{data} specifying the
variables/items.}

\item{ci.type}{character vector of length 1 specifying the type of confidence
interval to compute. The options are 1) "classic" is the Feldt et al.
(1987) procedure using only the mean covariance, 2) "delta" is the
Duhhacheck & Iacobucci (2004) procedure using the delta method of the
covariance matrix, or 3) "boot" is bootstrapped confidence intervals with
the method specified by \code{boot.ci.type}.}

\item{level}{double vector of length 1 with a value between 0 and 1
specifying what confidence level to use.}

\item{use}{character vector of length 1 specifying how to handle missing data
when computing the covariances. The options are 1) "pairwise.complete.obs",
2) "complete.obs", 3) "na.or.complete", 4) "all.obs", or 5) "everything".
See details of \code{\link{cov}}.}

\item{stats}{character vector specifying the additional statistical
information you could like related to cronbach's alpha. Options are: 1)
"std.alpha" = cronbach's alpha of the standardized variables/items, 2)
"G6(smc)" = Guttman's Lambda 6 reliability, 3) "average_r" = mean
correlation between the variables/items, 4) "median_r" = median correlation
between the variables/items, 5) "mean" = mean of the the score from
averaging the variables/items together, 6) "sd" = standard deviation of the
scores from averaging the variables/items together, 7) "nvrb" = number of
variables/items. The default is "average_r" and "nvrb".}

\item{R}{integer vector of length 1 specifying the number of bootstrapped
resamples to do. Only used when \code{ci.type} = "boot".}

\item{boot.ci.type}{character vector of length 1 specifying the type of
bootstrapped confidence interval to compute. The options are 1) "perc" for
the regular percentile method, 2) "bca" for bias-corrected and accelerated
percentile method, 3) "norm" for the normal method that uses the
bootstrapped standard error to construct symmetrical confidence intervals
with the classic formula around the bias-corrected estimate, and 4) "basic"
for the basic method. Note, "stud" for the studentized method is NOT an
option. See \code{\link[boot]{boot.ci}} as well as
\code{\link{confint2.boot}} for details.}
}
\value{
double vector containing Cronbach's alpha, it's standard error, and
  it's confidence interval, followed by any statistics requested via the
  \code{stats} argument.
}
\description{
\code{cronbach} computes Cronbach's alpha for a set of variables/items as an
estimate of reliability for a score. There are three different options for
confidence intervals. Missing data can be handled by either pairwise deletion
(\code{use} = "pairwise.complete.obs") or listwise deletion (\code{use} =
"complete.obs"). \code{cronbach} is a wrapper for the
\code{\link[psych]{alpha}} function in the \code{psych} package.
}
\details{
When \code{ci.type} = "classic" the confidence interval is based on the mean
covariance. It is the same as the confidence interval used by
\code{\link[psych]{alpha.ci}} (Feldt, Woodruff, & Salih, 1987). When
\code{ci.type} = "delta" the confidence interval is based on the delta method
of the covariance matrix. It is based on the standard error returned by
\code{\link[psych]{alpha}} (Duhachek & Iacobucci, 2004).
}
\examples{

tmp_nm <- c("A2","A3","A4","A5")
psych::alpha(psych::bfi[tmp_nm])[["total"]]
a <- suppressMessages(psych::alpha(attitude))[["total"]]["raw_alpha"]
a.ci <- psych::alpha.ci(a, n.obs = 30,
   n.var = 7, digits = 7) # n.var is optional and only needed to find r.bar
cronbach(data = psych::bfi, vrb.nm = c("A2","A3","A4","A5"), ci.type = "classic")
cronbach(data = psych::bfi, vrb.nm = c("A2","A3","A4","A5"), ci.type = "delta")
cronbach(data = psych::bfi, vrb.nm = c("A2","A3","A4","A5"), ci.type = "boot")
cronbach(data = psych::bfi, vrb.nm = c("A2","A3","A4","A5"), stats = NULL)

\dontrun{
cronbach(data = psych::bfi, vrb.nm = c("A2","A3","A4","A5"), ci.type = "boot",
   boot.ci.type = "bca") # will automatically convert to "perc" when "bca" fails
}

}
\references{
Feldt, L. S., Woodruff, D. J., & Salih, F. A. (1987). Statistical inference for
coefficient alpha. Applied Psychological Measurement (11) 93-103.

Duhachek, A. and Iacobucci, D. (2004). Alpha's standard error (ase): An accurate
and precise confidence interval estimate. Journal of Applied Psychology, 89(5):792-808.
}
\seealso{
\code{\link{cronbachs}}
   \code{\link{composite}}
}
