#' Model the Data
#'
#' Using the methodology generated by Muoko et. al (see README for
#' full citation), run the modelling with the JAGS sampler. This
#' function accepts a number of different arguments, but defaulted
#' arguments are assumed to be best for _most_ systems. Obviously,
#' if there is prior knowledge, this can be adjusted at the user's
#' own discretion.
#'
#' @param data data.frame The data to format
#' @param columns vector The columns to target
#' @param n_chains numeric Number of chains to run, default = 1
#' @param n_adapt_steps numeric Number of adapt steps to run, default = 500
#' @param n_burn numeric Number of draws to burn at the start, default = 1000
#' @param n_thin numeric Thinning factor for the draws, default = 1
#' @param n_steps numeric The total number of draws to run, default = 10,000
#' @param seed numeric The random seed to set
#' @param stan boolean If you would like to use the experimental Stan backend
#' @return List with all the appropriate things needed
#'         for modelling with JAGS
#' @export
#' @import rjags
#' @import rstan
#' @import stats
#'
#' @examples
#' data <- data.frame(list("fruit" = c(1, 2), "veg" = c(3, 4)))
#' acme_model(data, names(data))
acme_model <- function(
    data,
    columns,
    n_chains = 1,
    n_adapt_steps = 500,
    n_burn = 1000,
    n_thin = 1,
    n_steps = 10000,
    seed = 42,
    stan = FALSE) {
  # Check if the correct type
  stopifnot(is.numeric(n_chains))
  stopifnot(is.numeric(n_adapt_steps))
  stopifnot(is.numeric(n_burn))
  stopifnot(is.numeric(n_thin))
  stopifnot(is.numeric(n_steps))
  stopifnot(is.numeric(seed))

  # Check for implausible values
  stopifnot(!(n_chains < 1))
  stopifnot(!(n_chains > 8))
  stopifnot(!(n_adapt_steps < 1))
  stopifnot(!(n_burn < 1))
  stopifnot(!(n_thin < 1))
  stopifnot(!(n_thin > 8))
  stopifnot(!(n_steps < 1))

  set.seed(seed)

  if (stan) {
    modelling_data <- create_modelling_data(data, columns)
    modelling_data$sd_orig <- as.numeric(modelling_data$sd_orig[1, ])
    modelling_data$mean_orig <- as.numeric(modelling_data$mean_orig[1, ])
    model_path <- create_stan_model_string()

    stan_pre_model <- rstan::stan_model(model_path)

    mcmc_samples <- rstan::sampling(
      stan_pre_model,
      modelling_data,
      pars = c("sigma", "rho"),
      chains = n_chains,
      iter = n_steps,
      thin = n_thin,
      seed = seed,
      warmup = n_adapt_steps + n_burn
    )

    scovmat <- summary(mcmc_samples)
    return(list(
      covariance_matrix = scovmat,
      samples = mcmc_samples,
      model = stan_pre_model
    ))
  } else {
    modelling_data <- create_modelling_data(data, columns)
    model_path <- create_model_string()

    jags_model <- rjags::jags.model(
      model_path,
      modelling_data,
      n.chains = n_chains,
      n.adapt = n_adapt_steps
    )

    stats::update(jags_model, n.iter = n_burn)
    mcmc_samples <- rjags::coda.samples(
      jags_model,
      variable.names = c("sigma", "rho"),
      n.iter = n_steps / n_chains * n_thin,
      thin = n_thin,
      seed = seed
    )
    scovmat <- summary(mcmc_samples)
    return(list(
      covariance_matrix = scovmat,
      samples = mcmc_samples,
      model = jags_model
    ))
  }
}
