% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_pdb_file.R
\name{get_pdb_file}
\alias{get_pdb_file}
\title{Download and Process PDB Files from the RCSB Database}
\usage{
get_pdb_file(
  pdb_id,
  filetype = "cif",
  rm.insert = FALSE,
  rm.alt = TRUE,
  compression = TRUE,
  save = FALSE,
  path = NULL,
  verbosity = TRUE,
  download_base_url = DOWNLOAD_BASE_URL
)
}
\arguments{
\item{pdb_id}{A 4-character string specifying the PDB entry of interest (e.g., "1XYZ"). This identifier uniquely represents
a macromolecular structure within the PDB database.}

\item{filetype}{A string specifying the format of the file to be downloaded. The default is 'cif'. Supported file types include:
\describe{
  \item{'pdb'}{The older PDB file format, which provides atomic coordinates and metadata.}
  \item{'cif'}{The Crystallographic Information File (CIF) format, which is a newer standard replacing PDB files.}
  \item{'xml'}{An XML format file, providing structured data that can be easily parsed for various applications.}
  \item{'structfact'}{Structure factor files in CIF format, available for certain PDB entries, containing experimental
  data used to determine the structure.}
}}

\item{rm.insert}{Logical flag indicating whether to ignore PDB insertion codes. Default is FALSE. If TRUE, records with insertion
codes will be removed from the final data.}

\item{rm.alt}{Logical flag indicating whether to ignore alternate location indicators (ALT) in PDB files. Default is TRUE. If
TRUE, only the first alternate location is kept, and others are removed.}

\item{compression}{Logical flag indicating whether to download the file in a compressed format (e.g., .gz). Default is TRUE,
which is recommended for faster downloads, especially for CIF files.}

\item{save}{Logical flag indicating whether to save the downloaded file to a specified directory. Default is FALSE, which means
the file is processed and optionally saved, but not retained after processing unless specified.}

\item{path}{A string specifying the directory where the downloaded file should be saved. If NULL, the file is saved in a
temporary directory. If `save` is TRUE, this path is required.}

\item{verbosity}{A boolean flag indicating whether to print status messages during the function execution.}

\item{download_base_url}{A string representing the base URL for the PDB file retrieval. By default, this is set to the global constant \code{DOWNLOAD_BASE_URL}, but users can specify a different URL if needed.}
}
\value{
A list of class \code{"pdb"} containing the following components:
  \describe{
    \item{\code{atom}}{A data frame containing atomic coordinate data (ATOM and HETATM records). Each row corresponds to an
    atom, and each column to a specific record type (e.g., element, residue, chain).}
    \item{\code{xyz}}{A numeric matrix of class \code{"xyz"} containing the atomic coordinates from the ATOM and HETATM records.}
    \item{\code{calpha}}{A logical vector indicating whether each atom is a C-alpha atom (TRUE) or not (FALSE).}
    \item{\code{call}}{The matched call, storing the function call for reference.}
    \item{\code{path}}{The file path where the file was saved, if `save` was TRUE.}
  }
  The function handles errors and warnings for various edge cases, such as unsupported file types, failed downloads, or issues
  with reading the file.
}
\description{
The `get_pdb_file` function is a versatile tool designed to download Protein Data Bank (PDB) files from the RCSB database.
It supports various file formats such as 'pdb', 'cif', 'xml', and 'structfact', with options for file compression and handling
alternate locations (ALT) and insertion codes (INSERT) in PDB files. This function also provides the flexibility to save
the downloaded files to a specified directory or to a temporary directory for immediate use.
}
\details{
The `get_pdb_file` function is an essential tool for structural biologists and bioinformaticians who need to download and
process PDB files for further analysis. By providing options to handle alternate locations and insertion codes, this function
ensures that the data is clean and ready for downstream applications. Additionally, the ability to save files locally or
work with them in a temporary directory provides flexibility for various workflows. Error handling and informative messages
are included to guide the user in case of issues with file retrieval or processing.
}
\examples{
\donttest{
  # Download a CIF file and process it without saving
  pdb_file <- get_pdb_file(pdb_id = "4HHB", filetype = "cif")

  # Download a PDB file, save it, and remove alternate location records
  pdb_file <- get_pdb_file(pdb_id = "4HHB", filetype = "pdb", save = TRUE, path = tempdir())

  # Understanding the tertiary structure of proteins is
  # crucial for elucidating their functional mechanisms,
  # especially in the context of ligand binding, enzyme catalysis,
  # and protein-protein interactions.
  # The tertiary structure refers to the three-dimensional arrangement
  # of all atoms within a protein,
  # including its secondary structure elements like alpha helices
  # and beta sheets, and how these elements
  # are organized in space. Using the get_pdb_file function
  # to retrieve the PDB file and the r3dmol
  # package for visualization, researchers can gain insights
  # into the overall 3D structure of a protein.
  # The following example demonstrates how to visualize the
  # ltertiary structure of a protein using the
  # PDB entry 1XYZ:

  library(r3dmol)

  # Retrieve and parse a PDB structure
  pdb_path <- get_pdb_file("1XYZ", filetype = "pdb", save = TRUE)

  # Visualize the tertiary structure using r3dmol
  viewer <- r3dmol() \%>\%
    m_add_model(pdb_path$path, format = "pdb") \%>\%  # Load the PDB file
    m_set_style(style = m_style_cartoon()) \%>\%  # Cartoon representation
    m_zoom_to()

  # Display the molecular viewer
  viewer
  # In this example, the protein structure is represented
  # in a cartoon style, which is particularly
  # effective for visualizing the overall fold of the protein,
  # including the orientation and interaction
  # of its secondary structure elements.
  #. To further enhance the analysis,
  # it is often important to
  # highlight specific regions of interest,
  # such as potential ligand-binding sites.
  # These sites can be identified based on prior knowledge,
  # experimental data, or computational predictions.
  # The following code snippet demonstrates
  # how to highlight potential ligand-binding sites in the
  # protein structure:

  # Highlight potential ligand-binding sites
  # Note: Manually define residues of interest based
  # on prior knowledge or external analysis
  binding_sites <- c(45, 60, 85)  # Example residue numbers

  viewer <- viewer \%>\%
    m_set_style(
      sel = m_sel(resi = binding_sites),
      style = m_style_sphere(color = "red", radius = 1.5)
    )

  # Display the updated viewer with highlighted binding sites
  viewer

  # In this step, specific residues that are
  # hypothesized to participate in ligand binding are
  #highlighted using a spherical representation.
  # The residues are selected manually based on either
  # experimental data or computational predictions.
  # By highlighting these sites, researchers can
  # visually inspect the spatial relationship between
  # these residues and other parts of the protein,
  # which may provide insights into the
  # protein's functional mechanisms.

  # This visualization approach offers a powerful
  # way to explore and communicate the 3D structure
  # of proteins, making it easier to hypothesize about their function and
  # interaction with other molecules.

}
}
