% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rd2d.R
\name{rd2d}
\alias{rd2d}
\title{Two-Dimensional Local Polynomial Regression Discontinuity Design}
\usage{
rd2d(
  Y,
  X,
  t,
  b,
  h = NULL,
  deriv = c(0, 0),
  tangvec = NULL,
  p = 1,
  q = 2,
  kernel = c("tri", "triangular", "epa", "epanechnikov", "uni", "uniform", "gau",
    "gaussian"),
  kernel_type = c("prod", "rad"),
  vce = c("hc1", "hc0", "hc2", "hc3"),
  masspoints = c("check", "adjust", "off"),
  C = NULL,
  level = 95,
  cbands = TRUE,
  side = c("two", "left", "right"),
  repp = 1000,
  bwselect = c("mserd", "imserd", "msetwo", "imsetwo", "user provided"),
  method = c("dpi", "rot"),
  bwcheck = 50 + p + 1,
  scaleregul = 3,
  scalebiascrct = 1,
  stdvars = TRUE
)
}
\arguments{
\item{Y}{Dependent variable; a numeric vector of length \eqn{N}, where \eqn{N} is the sample size.}

\item{X}{Bivariate running variable (a.k.a score variable); a numeric matrix or data frame of dimension \eqn{N \times 2}, with each row \eqn{\mathbf{X}_i = (X_{1i}, X_{2i})}.}

\item{t}{Treatment indicator; a logical or binary vector indicating treatment assignment (\eqn{t_i = 1} if treated, \eqn{t_i = 0} otherwise).}

\item{b}{Evaluation points; a matrix or data frame specifying boundary points \eqn{\mathbf{b}_j = (b_{1j}, b_{2j})}, of dimension \eqn{J \times 2}.}

\item{h}{Bandwidths. Either a positive scalar (same bandwidth for all dimensions and groups), or a matrix/data frame of size \eqn{J \times 4}, corresponding to \eqn{h_{\text{control},1}},
\eqn{h_{\text{control},2}}, \eqn{h_{\text{treated},1}}, \eqn{h_{\text{treated},2}} at each evaluation point. If not specified, bandwidth \code{h} is computed by the companion command
\code{rdbw2d}. Default is \code{h = NULL}.}

\item{deriv}{The order of the derivatives of the regression functions to be estimated; a numeric vector of length 2 specifying the number of derivatives in each coordinate (e.g., \eqn{c(1,2)} corresponds to \eqn{\partial_1 \partial_2^2}).}

\item{tangvec}{Tangent vectors; a matrix or data frame of dimension \eqn{J \times 2} specifying directional derivatives. Overrides \code{deriv} if provided.}

\item{p}{Polynomial order for point estimation (\eqn{p = 1} by default).}

\item{q}{Polynomial order for robust confidence interval construction. Must satisfy \eqn{q \geq p}; default is \eqn{q = p + 1}.}

\item{kernel}{Kernel function to use. Options are \code{"unif"}, \code{"uniform"} (uniform), \code{"triag"}, \code{"triangular"} (triangular, default), and \code{"epan"}, \code{"epanechnikov"} (Epanechnikov).}

\item{kernel_type}{Kernel structure. Either \code{"prod"} for product kernels (default) or \code{"rad"} for radial kernels.}

\item{vce}{Variance-covariance estimation method. Options are:
\itemize{
\item \code{"hc0"}: heteroskedasticity-robust plug-in residual variance estimator without small-sample adjustment.
\item \code{"hc1"}: heteroskedasticity-robust plug-in residual variance estimator with HC1 small-sample adjustment (default).
\item \code{"hc2"}: heteroskedasticity-robust plug-in residual variance estimator with HC2 adjustment.
\item \code{"hc3"}: heteroskedasticity-robust plug-in residual variance estimator with HC3 adjustment.
}}

\item{masspoints}{Handling of mass points in the running variable. Options are:
\itemize{
\item \code{"check"}: detects presence of mass points and reports the number of unique observations (default).
\item \code{"adjust"}: adjusts preliminary bandwidths to ensure a minimum number of unique observations within each side of the cutoff.
\item \code{"off"}: ignores presence of mass points.
}}

\item{C}{Cluster ID variable used for cluster-robust variance estimation. Default is \code{C = NULL}.}

\item{level}{Nominal confidence level for intervals/bands, between 0 and 100 (default is 95).}

\item{cbands}{Logical. If \code{TRUE}, also compute uniform confidence bands (default is \code{FALSE}).}

\item{side}{Type of confidence interval. Options: \code{"two"} (two-sided, default), \code{"left"} (left tail), or \code{"right"} (right tail).}

\item{repp}{Number of repetitions for critical value simulation (used in uniform confidence bands). Default is 1000.}

\item{bwselect}{Bandwidth selection strategy. Options:
\itemize{
\item \code{"mserd"}. One common MSE-optimal bandwidth selector for the boundary RD treatment effect estimator for each evaluation point (default).
\item \code{"imserd"}. IMSE-optimal bandwidth selector for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"msetwo"}. Two different MSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator for each evaluation point.
\item \code{"imsetwo"}. Two IMSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"user provided"}. User-provided bandwidths. If \code{h} is not \code{NULL}, then \code{bwselect} is overwritten to \code{"user provided"}.
}}

\item{method}{Bandwidth selection method for bias estimator based on local polynomials. Either \code{"dpi"} (default) for data-driven plug-in MSE optimal bandwidth selector or \code{"rot"} for rule-of-thumb bandwidth selector.}

\item{bwcheck}{If a positive integer is provided, the preliminary bandwidth used in the calculations is enlarged so that at least \code{bwcheck} observations are used. If \code{masspoints == "adjust"}, ensure at least \code{bwcheck} unique observations are used. The program stops with “not enough observations” if sample size \eqn{N} < \code{bwcheck}. Default is \code{50 + p + 1}.}

\item{scaleregul}{Scaling factor for the regularization term in bandwidth selection. Default is 3.}

\item{scalebiascrct}{Scaling factor used for bias correction based on higher order expansions. Default is 1.}

\item{stdvars}{Logical. If TRUE, the running variables \eqn{X_{1i}} and \eqn{X_{2i}} are standardized before computing the bandwidths. Default is \code{TRUE}. Standardization only affects automatic bandwidth selection if bandwidths are not manually provided via \code{h}.}
}
\value{
An object of class \code{"rd2d"}, a list with components:
\describe{
\item{\code{results}}{A data frame with point estimates, variances, p-values, confidence intervals, confidence bands, bandwidths and effective sample size at each evaluation point.
\describe{
\item{\code{b1}, \code{b2}}{First and second coordinate of evaluation points \eqn{\mathbf{b} = (b_1,b_2)}.}
\item{\code{Est.p}}{Point estimate of \eqn{\widehat{\tau}_p(\mathbf{b})}.}
\item{\code{Se.p}}{Standard error of \eqn{\widehat{\tau}_p(\mathbf{b})}.}
\item{\code{Est.q}}{Bias-corrected point estimate of \eqn{\widehat{\tau}_q(\mathbf{b})}.}
\item{\code{Se.q}}{Standard error of bias-corrected estimate \eqn{\widehat{\tau}_q(\mathbf{b})}.}
\item{\code{p-value}}{P-value based on t-statistic with bias-corrected estimate.}
\item{\code{CI.lower}, \code{CI.upper}}{Pointwise confidence intervals.}
\item{\code{CB.lower}, \code{CB.upper}}{Uniform confidence bands if computed.}
\item{\code{h01}, \code{h02}, \code{h11}, \code{h12}}{Bandwidths used in each coordinate and group. The four columns correspond to \eqn{h_{\text{control},1}},
\eqn{h_{\text{control},2}}, \eqn{h_{\text{treated},1}}, \eqn{h_{\text{treated},2}} respectively.}
\item{\code{Nh0}, \code{Nh1}}{Effective sample size on each side of the cutoff.}
}
}
\item{\code{results.A0}}{Same structure as \code{results} but for control group outcomes.}
\item{\code{results.A1}}{Same structure as \code{results} but for treated group outcomes.}
\item{\code{cov.q}}{Covariance matrix for bias-corrected estimates \eqn{\widehat{\tau}_q(\mathbf{b})} for all point evaluations \eqn{\mathbf{b}}.}
\item{\code{opt}}{List of options used in the function call.}
\item{\code{rdmodel}}{String label for the RD model.}
}
}
\description{
\code{rd2d} implements bivariate local polynomial boundary regression discontinuity (RD) point estimators with robust bias-corrected pointwise confidence intervals and
uniform confidence bands, developed in Cattaneo, Titiunik and Yu (2025a) with a companion software article Cattaneo, Titiunik and Yu (2025b). For robust bias-correction, see Calonico, Cattaneo, Titiunik (2014).

Companion commands are: \code{rdbw2d} for data-driven bandwidth selection.

For other packages of RD designs, visit
\url{https://rdpackages.github.io/}
}
\examples{
# Simulated example
set.seed(123)
n <- 5000
X1 <- rnorm(n)
X2 <- rnorm(n)
t <- as.numeric(X1 > 0)
Y <- 3 + 2 * X1 + 1.5 * X2 + t + rnorm(n)
X <- cbind(X1, X2)
b <- matrix(c(0, 0, 0, 1), ncol = 2)

# Estimate treatment effect using rd2d
result <- rd2d(Y, X, t, b, cbands = TRUE)
print(result)
summary(result)
}
\references{
\itemize{
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_BoundaryRD.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025a).}
Estimation and Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_rd2d.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025b).}
rd2d: Causal Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Calonico-Cattaneo-Titiunik_2014_ECMA.pdf}{Calonico, S., Cattaneo, M. D., Titiunik, R. (2014)}
Robust Nonparametric Confidence Intervals for Regression-Discontinuity Designs}
}
}
\seealso{
\code{\link{rdbw2d}}, \code{\link{print.rd2d}}, \code{\link{summary.rd2d}}
}
\author{
Matias D. Cattaneo, Princeton University. \email{cattaneo@princeton.edu} \cr
Rocío Titiunik, Princeton University. \email{titiunik@princeton.edu} \cr
Ruiqi Rae Yu, Princeton University. \email{rae.yu@princeton.edu}
}
