% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/power.cmh.test.r
\name{power.cmh.test}
\alias{power.cmh.test}
\title{Power and sample size calculation for the Cochran-Mantel-Haenszel test}
\usage{
power.cmh.test(
  p1 = NULL,
  p2 = NULL,
  theta = NULL,
  N = NULL,
  sig.level = 0.05,
  power = 0.8,
  alternative = c("two.sided", "less", "greater"),
  s = 0.5,
  t = 1/J,
  correct = TRUE
)
}
\arguments{
\item{p1}{Vector of proportions of the \emph{J} case groups.}

\item{p2}{Vector of proportions of the \emph{J} control groups.}

\item{theta}{Vector of odds ratios relating to the \emph{J} 2 x 2 tables.}

\item{N}{Total number of subjects.}

\item{sig.level}{Significance level (Type I error probability).}

\item{power}{Power of test (1 minus Type II error probability).}

\item{alternative}{Two- or one-sided test. If one-sided, the direction of the
association must be defined (less than 1 or greater than 1). Can be
abbreviated.}

\item{s}{Proportion (weight) of case versus control in \emph{J} stratum.}

\item{t}{Proportion (weight) of total number of cases of \emph{J} stratum.}

\item{correct}{Logical indicating whether to apply continuity correction.}
}
\value{
An object of class \code{"power.cmh"}: a list of the original arguments and
the calculated sample size or power. Also included are vectors of n's per
each group, an indicator or whether continuity correction was used, the
original function call, and \code{N.effective}.

The vectors of n's per each group, \code{n1} and \code{n2}, are the
fractional n's required to achieve a final total N specified by the
calculation while satisfying the constraints of \code{s} and \code{t}.
However, the effective N, given the requirement of cell counts populated by
whole numbers is provided by \code{N.effective}. By default, the print method
is set to \code{n.frac = FALSE}, which will round each cell n up to the
nearest whole number.
}
\description{
Compute the post-hoc power or required number of subjects for the
Cochran-Mantel-Haenszel test for association in \emph{J} stratified 2 x 2
tables.
}
\details{
This sample size calculation is based on the derivations described in the
Woolson \emph{et al.} (1986). It is designed for case-control studies where
one margin is fixed. The method is "based on the Cochran-Mantel-Haenszel
statistic expressed as a weighted difference in binomial proportions."

Continuity corrected sample size is described in Nam's 1992 paper. This uses
the weighted binomial sample size calculation described in Woolson \emph{et
al.} (1986) but is enhanced for use with the continuity corrected Cochran's
test.

Power calculations are based on the writings of Wittes and Wallenstein
(1987). They are functionally equivalent to the derivations of the sample
size calculation described by Woolson and others and Nam, but have slightly
added precision.

Terminology and symbolic conventions are borrowed from Woolson \emph{et al.}
(1986). The \code{p1} group is dubbed the \emph{Case} group and \code{p2}
group is called the \emph{Control} group.
}
\section{Arguments}{


To calculate \strong{power}, the \code{power} parameter must be set to
\code{NULL}. To calculate \strong{sample size}, the \code{N} parameter must
be set to \code{NULL}.

The \code{J} number of groups will be inferred by the maximum length of
\code{p1}, \code{p2}, or \code{theta}.

Effect size must be specified using one of the following combinations of
arguments.
\itemize{
  \item Both case and control proportion vectors, ex.,
    \itemize{\item \code{p1} and \code{p2} with \code{theta = NULL}.}
  \item One proportion vector and an effect size, ex.,
    \itemize{
      \item \code{p1} and \code{theta} with \code{p2 = NULL}, \strong{or}
      \item \code{p2} and \code{theta} with \code{p1 = NULL}.
    }
}
}

\examples{
# From "Sample size determination for case-control studies and the comparison
# of stratified and unstratified analyses", (Nam 1992). See references.

# Uncorrected sample size estimate first introduced
# by Woolson and others in 1986
sample_size_uncorrected <- power.cmh.test(
  p2 = c(0.75,0.70,0.65,0.60),
  theta = 3,
  power = 0.9,
  t = c(0.10,0.40,0.35,0.15),
  alternative = "greater",
  correct = FALSE
)

print(sample_size_uncorrected, detail = FALSE)

# We see that the N is 171, the same as calculated by Nam
sample_size_uncorrected$N


# Continuity corrected sample size estimate added by Nam
sample_size_corrected <- power.cmh.test(
  p2 = c(0.75,0.70,0.65,0.60),
  theta = 3,
  power = 0.9,
  t = c(0.10,0.40,0.35,0.15),
  alternative = "greater",
  correct = TRUE
)

print(sample_size_corrected, n.frac = TRUE)

# We see that the N is indeed equal to that which is reported in the paper
sample_size_corrected$N

}
\references{
Gail, M. (1973). "The determination of sample sizes for trials involving
several 2 x 2 tables."
\emph{Journal of Chronic Disease} \strong{26}: 669-673.

Munoz, A. and B. Rosner. (1984). "Power and sample size for a collection of 2
x 2 tables." \emph{Biometrics} \strong{40}: 995-1004.

Nam, J. (1992). "Sample size determination for case-control studies and the
comparison of stratified and unstratified analyses."
\emph{Biometrics} \strong{48}: 389-395.

Wittes, J. and S. Wallenstein. (1987). "The power of the Mantel-Haenszel
test." \emph{Journal of the American Statistical Association} \strong{82}:
1104-1109.

Woolson, R. F., Bean, J. A., and P. B. Rojas. (1986).
"Sample size for case-control studies using Cochran's statistic."
\emph{Biometrics} \strong{42}: 927-932.
}
\seealso{
\link[stats]{power.prop.test},
\link[stats]{mantelhaen.test},
\link[DescTools]{BreslowDayTest}
}
\author{
Paul W. Egeler, M.S.
}
