## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

library(sdtm.oak)
library(admiraldev)
library(rlang)
library(dplyr, warn.conflicts = FALSE)

## ----eval=TRUE----------------------------------------------------------------
dm_raw <- pharmaverseraw::dm_raw
ds_raw <- pharmaverseraw::ds_raw
ec_raw <- pharmaverseraw::ec_raw

## ----eval=TRUE, echo=FALSE----------------------------------------------------
sdtm.oak:::dataset_oak_vignette(
  dm_raw,
  display_vars = exprs(
    PATNUM, IT.AGE, IT.SEX, IT.ETHNIC, IT.RACE, COUNTRY, PLANNED_ARM,
    PLANNED_ARMCD, ACTUAL_ARM, ACTUAL_ARMCD, COL_DT, IC_DT
  )
)

sdtm.oak:::dataset_oak_vignette(
  ds_raw,
  display_vars = exprs(
    PATNUM, INSTANCE, IT.DSTERM, OTHERSP, IT.DSSTDAT, DEATHDT
  )
)

sdtm.oak:::dataset_oak_vignette(
  ec_raw,
  display_vars = exprs(
    PATNUM, VISITNAME, IT.ECSTDAT, IT.ECENDAT
  )
)

## ----eval=TRUE----------------------------------------------------------------
dm_raw <- dm_raw %>%
  generate_oak_id_vars(
    pat_var = "PATNUM",
    raw_src = "dm_raw"
  )

ds_raw <- ds_raw %>%
  generate_oak_id_vars(
    pat_var = "PATNUM",
    raw_src = "ds_raw"
  )

ec_raw <- ec_raw %>%
  generate_oak_id_vars(
    pat_var = "PATNUM",
    raw_src = "ec_raw"
  )

## ----eval=TRUE, echo=FALSE----------------------------------------------------
sdtm.oak:::dataset_oak_vignette(
  dm_raw,
  display_vars = exprs(
    oak_id, raw_source, patient_number, PATNUM, IT.SEX, IT.ETHNIC, IT.RACE,
    COUNTRY, PLANNED_ARM
  )
)

## ----eval=TRUE----------------------------------------------------------------
study_ct <- read.csv(system.file("raw_data/sdtm_ct.csv",
  package = "sdtm.oak"
))

## ----eval=TRUE, echo=FALSE----------------------------------------------------
sdtm.oak:::dataset_oak_vignette(
  study_ct,
  display_vars = exprs(
    codelist_code, term_code, term_value, collected_value, term_preferred_term,
    term_synonyms
  )
)

## ----eval=TRUE----------------------------------------------------------------
ref_date_conf_df <- tibble::tribble(
  ~raw_dataset_name, ~date_var,     ~time_var,      ~dformat,      ~tformat, ~sdtm_var_name,
  "ec_raw",       "IT.ECSTDAT", NA_character_, "dd-mmm-yyyy", NA_character_,     "RFXSTDTC",
  "ec_raw",       "IT.ECENDAT", NA_character_, "dd-mmm-yyyy", NA_character_,     "RFXENDTC",
  "ds_raw",       "IT.DSSTDAT", NA_character_,  "mm-dd-yyyy", NA_character_,      "RFSTDTC",
  "ec_raw",       "IT.ECENDAT", NA_character_, "dd-mmm-yyyy", NA_character_,      "RFENDTC",
  "dm_raw",            "IC_DT", NA_character_,  "mm/dd/yyyy", NA_character_,      "RFICDTC",
  "ds_raw",          "DSDTCOL",     "DSTMCOL",  "mm-dd-yyyy",         "H:M",     "RFPENDTC",
  "ds_raw",          "DEATHDT", NA_character_,  "mm/dd/yyyy", NA_character_,       "DTHDTC"
)

## ----eval=TRUE, echo=FALSE----------------------------------------------------
sdtm.oak:::dataset_oak_vignette(
  ref_date_conf_df,
  display_vars = exprs(
    raw_dataset_name, date_var, time_var, dformat, tformat, sdtm_var_name
  )
)

## ----eval=TRUE----------------------------------------------------------------
dm <-
  # Map AGE using assign_no_ct
  assign_no_ct(
    raw_dat = dm_raw,
    raw_var = "IT.AGE",
    tgt_var = "AGE",
    id_vars = oak_id_vars()
  ) %>%
  # Map AGEU using assign_ct
  hardcode_ct(
    raw_dat = dm_raw,
    raw_var = "IT.AGE",
    tgt_var = "AGEU",
    tgt_val = "Year",
    ct_spec = study_ct,
    ct_clst = "C66781",
    id_vars = oak_id_vars()
  ) %>%
  # Map SEX using assign_no_ct
  assign_ct(
    raw_dat = dm_raw,
    raw_var = "IT.SEX",
    tgt_var = "SEX",
    ct_spec = study_ct,
    ct_clst = "C66731",
    id_vars = oak_id_vars()
  ) %>%
  # Map ETHNIC using assign_no_ct
  assign_ct(
    raw_dat = dm_raw,
    raw_var = "IT.ETHNIC",
    tgt_var = "ETHNIC",
    ct_spec = study_ct,
    ct_clst = "C66790",
    id_vars = oak_id_vars()
  ) %>%
  # Map RACE using assign_no_ct
  assign_ct(
    raw_dat = dm_raw,
    raw_var = "IT.RACE",
    tgt_var = "RACE",
    ct_spec = study_ct,
    ct_clst = "C74457",
    id_vars = oak_id_vars()
  ) %>%
  # Map ARM using assign_no_ct
  assign_ct(
    raw_dat = dm_raw,
    raw_var = "PLANNED_ARM",
    tgt_var = "ARM",
    ct_spec = study_ct,
    ct_clst = "ARM",
    id_vars = oak_id_vars()
  ) %>%
  # Map ARMCD using assign_no_ct
  assign_no_ct(
    raw_dat = dm_raw,
    raw_var = "PLANNED_ARMCD",
    tgt_var = "ARMCD",
    id_vars = oak_id_vars()
  ) %>%
  # Map ACTARM using assign_no_ct
  assign_ct(
    raw_dat = dm_raw,
    raw_var = "ACTUAL_ARM",
    tgt_var = "ACTARM",
    ct_spec = study_ct,
    ct_clst = "ARM",
    id_vars = oak_id_vars()
  ) %>%
  # Map ACTARMCD using assign_no_ct
  assign_no_ct(
    raw_dat = dm_raw,
    raw_var = "ACTUAL_ARMCD",
    tgt_var = "ACTARMCD",
    id_vars = oak_id_vars()
  )

## ----eval=TRUE, echo=FALSE----------------------------------------------------
sdtm.oak:::dataset_oak_vignette(
  dm,
  display_vars = exprs(
    oak_id, raw_source, patient_number, AGE, AGEU, SEX, ETHNIC, RACE,
    ARM, ARMCD, ACTARM, ACTARMCD
  )
)

## ----eval=TRUE----------------------------------------------------------------
dm <- dm %>%
  mutate(
    STUDYID = dm_raw$STUDY,
    DOMAIN = "DM",
    SUBJID = substr(dm_raw$PATNUM, 5, 8),
    USUBJID = paste0("01-", dm_raw$PATNUM),
    COUNTRY = dm_raw$COUNTRY
  ) %>%
  # Map DMDTC using assign_datetime
  assign_datetime(
    raw_dat = dm_raw,
    raw_var = "COL_DT",
    tgt_var = "DMDTC",
    raw_fmt = c("m/d/y"),
    id_vars = oak_id_vars()
  )

## ----eval=TRUE----------------------------------------------------------------
dm <- dm %>%
  # Map DTHDTC using oak_cal_ref_dates
  oak_cal_ref_dates(
    ds_in = .,
    der_var = "DTHDTC",
    min_max = "min",
    ref_date_config_df = ref_date_conf_df,
    raw_source = list(
      ec_raw = ec_raw,
      ds_raw = ds_raw,
      dm_raw = dm_raw
    )
  )

## ----eval=TRUE----------------------------------------------------------------
dm <- dm %>%
  # Derive RFSTDTC using oak_cal_ref_dates
  oak_cal_ref_dates(
    ds_in = .,
    der_var = "RFSTDTC",
    min_max = "min",
    ref_date_config_df = ref_date_conf_df,
    raw_source = list(
      ec_raw = ec_raw,
      ds_raw = ds_raw,
      dm_raw = dm_raw
    )
  )

## ----eval=TRUE----------------------------------------------------------------
dm <- dm %>%
  # Derive RFENDTC using oak_cal_ref_dates
  oak_cal_ref_dates(
    ds_in = .,
    der_var = "RFENDTC",
    min_max = "max",
    ref_date_config_df = ref_date_conf_df,
    raw_source = list(
      ec_raw = ec_raw,
      ds_raw = ds_raw,
      dm_raw = dm_raw
    )
  )

