% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/standardizedSolution_boot_ci.R
\name{standardizedSolution_boot}
\alias{standardizedSolution_boot}
\title{Bootstrap CIs for Standardized
Solution}
\usage{
standardizedSolution_boot(
  object,
  level = 0.95,
  type = "std.all",
  boot_delta_ratio = FALSE,
  boot_ci_type = c("perc", "bc", "bca.simple"),
  save_boot_est_std = TRUE,
  boot_pvalue = TRUE,
  boot_pvalue_min_size = 1000,
  ...
)
}
\arguments{
\item{object}{A 'lavaan'-class
object, fitted with 'se = "boot"'.}

\item{level}{The level of confidence
of the confidence intervals. Default
is .95.}

\item{type}{The type of standard
estimates. The same argument of
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}},
and support all values supported by
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}.
Default is \code{"std.all"}.}

\item{boot_delta_ratio}{The ratio of
(a) the distance of the bootstrap
confidence limit from the point
estimate to (b) the distance of the
delta-method limit from the point
estimate. Default is \code{FALSE}.}

\item{boot_ci_type}{The type of the
bootstrapping confidence intervals.
Support percentile confidence intervals
(\code{"perc"}, the default) and
bias-corrected confidence intervals
(\code{"bc"} or \code{"bca.simple"}).}

\item{save_boot_est_std}{Whether the
bootstrap estimates of the
standardized solution are saved. If
saved, they will be stored in the
attribute \code{boot_est_std}. Default is
\code{TRUE}.}

\item{boot_pvalue}{Whether asymmetric
bootstrap \emph{p}-values are computed.
Default is \code{TRUE}.}

\item{boot_pvalue_min_size}{Integer.
The asymmetric bootstrap \emph{p}-values
will be computed only if the number
of valid bootstrap estimates is at
least this value. Otherwise, \code{NA}
will be returned. If the number of
valid bootstrap samples is less than
this value, then \code{boot_pvalue} will
be set to \code{FALSE}.}

\item{...}{Other arguments to be
passed to
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}.}
}
\value{
The output of
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}},
with bootstrap confidence intervals
appended to the right, with class
set to \code{sbt_std_boot}. It has
a print method
(\code{\link[=print.sbt_std_boot]{print.sbt_std_boot()}}) that
can be used to print the standardized
solution in a format similar to
that of the printout of
the \code{\link[=summary]{summary()}} of a \link[lavaan:lavaan-class]{lavaan::lavaan} object.
}
\description{
Functions for forming
bootstrap confidence intervals
for the standardized solution.
}
\details{
\code{\link[=standardizedSolution_boot]{standardizedSolution_boot()}}
receives a
\link[lavaan:lavaan-class]{lavaan::lavaan} object fitted
with bootstrapping standard errors
requested and forms the confidence
intervals for the standardized
solution.

It works by calling
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}
with the bootstrap estimates
of free parameters in each bootstrap sample
to compute the standardized estimates
in each sample.

Alternative, call \code{\link[=store_boot]{store_boot()}} to
computes and store bootstrap estimates
of the standardized solution.
This function will then retrieve them,
even if \code{se} was not set to
\code{"boot"} or \code{"bootstrap"} when fitting
the model.
\subsection{Bootstrap Confidence Intervals}{

It supports percentile and
bias-corrected bootstrap confidence
intervals.
}

\subsection{Bootstrap Standard Errors}{

The standard errors are the
standard deviation of the bootstrap
estimates, which can be different from
the delta-method standard errors.
}

\subsection{Bootstrap Asymmetric \emph{p}-Values}{

If percentile bootstrap confidence
interval is requested, asymmetric
bootstrap \emph{p}-values are also
computed, using the method presented
in Asparouhov and Muthén (2021).
}
}
\examples{

library(lavaan)
set.seed(5478374)
n <- 50
x <- runif(n) - .5
m <- .40 * x + rnorm(n, 0, sqrt(1 - .40))
y <- .30 * m + rnorm(n, 0, sqrt(1 - .30))
dat <- data.frame(x = x, y = y, m = m)
model <-
'
m ~ a*x
y ~ b*m
ab := a*b
'

# Should set bootstrap to at least 2000 in real studies
fit <- sem(model, data = dat, fixed.x = FALSE,
           se = "boot",
           bootstrap = 100)
summary(fit)

std <- standardizedSolution_boot(fit)
std

# Print in a friendly format with only standardized solution
print(std, output = "text")

# Print in a friendly format with both unstandardized
# and standardized solution
print(std, output = "text", standardized_only = FALSE)

# hist_qq_boot() can be used to examine the bootstrap estimates
# of a parameter
hist_qq_boot(std, param = "ab")

# scatter_boot() can be used to examine the bootstrap estimates
# of two or more parameters
scatter_boot(std, params = c("ab", "a", "b"))

}
\references{
Asparouhov, A., & Muthén, B. (2021). Bootstrap p-value computation.
Retrieved from https://www.statmodel.com/download/FAQ-Bootstrap\%20-\%20Pvalue.pdf
}
\seealso{
\code{\link[lavaan:standardizedSolution]{lavaan::standardizedSolution()}}, \code{\link[=store_boot]{store_boot()}}
}
\author{
Shu Fai Cheung
\url{https://orcid.org/0000-0002-9871-9448}.
Originally proposed in an issue at GitHub
\url{https://github.com/simsem/semTools/issues/101#issue-1021974657},
inspired by a discussion at
the Google group for lavaan
\url{https://groups.google.com/g/lavaan/c/qQBXSz5cd0o/m/R8YT5HxNAgAJ}.
\code{\link[boot:boot.ci]{boot::boot.ci()}} is used to form the
percentile confidence intervals in
this version.
}
