% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semmcci-mc-std.R
\name{MCStd}
\alias{MCStd}
\title{Standardized Monte Carlo Confidence Intervals}
\usage{
MCStd(mc, alpha = c(0.001, 0.01, 0.05))
}
\arguments{
\item{mc}{Output of the \code{\link[=MC]{MC()}} or \code{\link[=MCMI]{MCMI()}} function.}

\item{alpha}{Numeric vector.
Significance level \eqn{\alpha}.}
}
\value{
Returns an object of class \code{semmcci} which is
a list with the following elements:
\describe{
\item{call}{Function call.}
\item{args}{List of function arguments.}
\item{thetahat}{Parameter estimates \eqn{\hat{\theta}}.}
\item{thetahatstar}{Sampling distribution of parameter estimates
\eqn{\hat{\theta}^{\ast}}.}
\item{fun}{Function used ("MCStd").}
}
}
\description{
Calculates standardized Monte Carlo confidence intervals
for free and defined parameters.
}
\details{
The empirical sampling distribution
of parameter estimates from the argument \code{mc} is standardized,
that is, each randomly generated vector of parameters is standardized.
Defined parameters are computed from the standardized component parameters.
Confidence intervals are generated
using the standardized empirical sampling distribution.
}
\examples{
library(semmcci)
library(lavaan)

# Data ---------------------------------------------------------------------
data("Tal.Or", package = "psych")
df <- mice::ampute(Tal.Or)$amp

# Monte Carlo --------------------------------------------------------------
## Fit Model in lavaan -----------------------------------------------------
model <- "
  reaction ~ cp * cond + b * pmi
  pmi ~ a * cond
  cond ~~ cond
  indirect := a * b
  direct := cp
  total := cp + (a * b)
"
fit <- sem(data = df, model = model, missing = "fiml")

## MC() --------------------------------------------------------------------
unstd <- MC(
  fit,
  R = 5L, # use a large value e.g., 20000L for actual research
  alpha = 0.05
)

## Standardized Monte Carlo ------------------------------------------------
MCStd(unstd, alpha = 0.05)

# Monte Carlo (Multiple Imputation) ----------------------------------------
## Multiple Imputation -----------------------------------------------------
mi <- mice::mice(
  data = df,
  print = FALSE,
  m = 5L, # use a large value e.g., 100L for actual research,
  seed = 42
)

## Fit Model in lavaan -----------------------------------------------------
fit <- sem(data = df, model = model) # use default listwise deletion

## MCMI() ------------------------------------------------------------------
unstd <- MCMI(
  fit,
  mi = mi,
  R = 5L, # use a large value e.g., 20000L for actual research
  alpha = 0.05
)

## Standardized Monte Carlo ------------------------------------------------
MCStd(unstd, alpha = 0.05)

}
\references{
Pesigan, I. J. A., & Cheung, S. F. (2024).
Monte Carlo confidence intervals for the indirect effect with missing data.
\emph{Behavior Research Methods}.
\doi{10.3758/s13428-023-02114-4}
}
\seealso{
Other Monte Carlo in Structural Equation Modeling Functions: 
\code{\link{Func}()},
\code{\link{MC}()},
\code{\link{MCFunc}()},
\code{\link{MCGeneric}()},
\code{\link{MCMI}()}
}
\author{
Ivan Jacob Agaloos Pesigan
}
\concept{Monte Carlo in Structural Equation Modeling Functions}
\keyword{mc}
\keyword{semmcci}
