% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotman.R
\name{plotman}
\alias{plotman}
\title{Second-Generation p-Value Plotting}
\usage{
plotman(
  est.lo,
  est.hi,
  null.lo,
  null.hi,
  set.order = NA,
  x.show = NA,
  type = "delta-gap",
  p.values = NA,
  ref.lines = NA,
  null.pt = NA,
  int.col = c("cornflowerblue", "firebrick3", "darkslateblue"),
  int.pch = 16,
  int.cex = 0.4,
  title.lab = NA,
  x.lab = "Position (by set.order)",
  y.lab = "Outcome label",
  legend.on = TRUE
)
}
\arguments{
\item{est.lo}{A numeric vector of lower bounds of interval estimates. Must be of same length as \code{est.hi}.}

\item{est.hi}{A numeric vector of upper bounds of interval estimates.  Must be of same length as \code{est.lo}.}

\item{null.lo}{A scalar representing the lower bound of the null interval hypothesis (indifference zone). Value must be finite.}

\item{null.hi}{A scalar representing the upper bound of the null interval hypothesis (indifference zone). Value must be finite.}

\item{set.order}{A numeric vector giving the desired order along the x-axis. Alternatively, if \code{set.order} is set to \code{"sgpv"}, the second-generation \emph{p}-value ranking is used. The default option is \code{NA}, which uses the original input ordering.}

\item{x.show}{A numeric scalar representing the maximum ranking on the x-axis that is displayed. Default is to display all rankings.}

\item{type}{A string specifying the desired Manhattan-style plot to be graphed. This argument specifies the variable on the y-axis. If \code{type = "delta-gap"}, the delta-gaps are ranked. If \code{type = "p-value"}, the classic p-values are ranked. If \code{type = "comparison"}, the classic p-values are ranked by SGPV. Default is \code{type = "delta-gap"}.}

\item{p.values}{A numeric vector giving the classic \emph{p}-values. This is required when \code{type = "p-value"} or \code{type = "comparison"}, and is not required when \code{type = "delta-gap"}. The \code{p.values} input may be any desired transformation of the p-values. For example, if the desired transformation is \eqn{-log10(p-value)} as in a traditional Manhattan plot, the \eqn{-log10(p-values)} should be provided for \code{p.values}. The corresponding x or y axis label(s) should be updated to reflect any transformations.}

\item{ref.lines}{A numeric scalar or vector giving the points on the y-axis at which to add a horizontal reference line. For example, if \code{p.values} is set to \eqn{-log10(p-values)} and the type of plot selected shows the (transformed) p-values on the y-axis, possible locations for the reference lines could be at the \eqn{-log10(0.05)}, \eqn{-log10(Bonferroni)} and \eqn{-log10(FDR)} significance levels.}

\item{null.pt}{An optional numeric scalar representing a point null hypothesis. Default is \code{NA}.}

\item{int.col}{Vector of length three specifing the colors of the points according to SGPV result. The first color option corresponds to the \eqn{SGPV = 0} results, the second color option corresponds to the \eqn{0 < SGPV < 1} results, and the third color option corresponds to the \eqn{SGPV = 1} results. Default is \code{int.col = c("cornflowerblue","firebrick3","darkslateblue")}.}

\item{int.pch}{Plotting symbol for points. Default is \code{16} for small points.}

\item{int.cex}{Size of plotting symbol for points. Default is \code{0.4}.}

\item{title.lab}{Title text.}

\item{x.lab}{A title for the x-axis. Default is the generic \code{"Position (by set.order)"}.}

\item{y.lab}{A title for the y-axis. Default is the generic \code{"Outcome label"}.}

\item{legend.on}{Toggle for plotting the legend. Default is \code{TRUE}.}
}
\description{
This function displays a modified Manhattan-style plot colored according to second-generation p-value status. There are several variations of this plot that can be made depending upon user input for \code{type} as well as the \code{set.order} and \code{x.show} options. These plots allow the user to visualize the overall result of a large-scale analysis succintly and to visually assess the differences in the results using second-generation p-value techniques as opposed to classical p-value techniques.
}
\details{
Use \code{set.order} to provide the classical p-value ranking. For example, if \code{pvalue.vector} is a vector of classical p-values, then set \code{set.order=order(pvalue.vector)} to sort the x-axis according to p-value rank.

Use \code{type} and \code{p.values} to provide the \eqn{-log10(p-values)} for the y-axis. For example, if \code{pvalue.vector} is a vector of classical p-values, then set \code{type="p-value"} (or \code{type="comparison"}) and \code{p.values=-log10(pvalue.vector)} to set the y-axis. Then, set the y-axis title to something like \code{y.lab="-log10(p)"}.
}
\examples{


#  Use leukstats data
data(leukstats)

# ID number on the x-axis, delta-gap on the y-axis, using an interval null hypothesis of
# (-0.3, 0.3) for the log mean difference in expression levels (fold change).
plotman(est.lo=leukstats$ci.lo, est.hi=leukstats$ci.hi,
       null.lo=-0.3, null.hi=0.3,
       set.order=NA,
       type="delta-gap",
       ref.lines=NA,
       int.pch=16, int.cex=0.4,
       title.lab="Leukemia Example",
       y.lab="Delta-gap",
       x.lab="Position (ID)",
       legend.on=TRUE)

# ID number on the x-axis, -log10(classical p-value) on the y-axis, using an interval
# null hypothesis of (-0.3, 0.3) for the log mean difference in expression levels
# (fold change).
plotman(est.lo=leukstats$ci.lo, est.hi=leukstats$ci.hi,
       null.lo=-0.3, null.hi=0.3,
       set.order=NA,
       type="p-value",
       p.values=-log10(leukstats$p.value),
       ref.lines=-log10(0.05),
       int.pch=16, int.cex=0.4,
       title.lab="Leukemia Example",
       y.lab=expression("-log"[10]*"(p-value)"),
       x.lab="Position (ID)",
       legend.on=TRUE)

# Second-generation p-value (SGPV) on the x-axis, -log10(classical p-value) on the
# y-axis, using an interval null hypothesis of (-0.3, 0.3) for the log mean difference
# in expression levels (fold change).
plotman(est.lo=leukstats$ci.lo, est.hi=leukstats$ci.hi,
       null.lo=-0.3, null.hi=0.3,
       set.order="sgpv",
       type="comparison",
       p.values=-log10(leukstats$p.value),
       ref.lines=c(-log10(0.05), -log10(0.001)),
       int.pch=16, int.cex=0.4,
       title.lab="Leukemia Example",
       y.lab=expression("-log"[10]*"(p-value)"),
       x.lab="Second-generation p-value ranking",
       legend.on=TRUE)

}
\references{
Blume JD, Greevy RA Jr., Welty VF, Smith JR, Dupont WD (2019). An Introduction to Second-generation \emph{p}-values. \emph{The American Statistician}. 73:sup1, 157-167, DOI: https://doi.org/10.1080/00031305.2018.1537893

Blume JD, D’Agostino McGowan L, Dupont WD, Greevy RA Jr. (2018). Second-generation \emph{p}-values: Improved rigor, reproducibility, & transparency in statistical analyses. \emph{PLoS ONE} 13(3): e0188299. https://doi.org/10.1371/journal.pone.0188299
}
\seealso{
\code{\link{sgpvalue}, \link{plotsgpv}, \link{sgpower}, \link{plotsgpower}}
}
