% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read_size_regression.R
\name{read_size_regression}
\alias{read_size_regression}
\title{Reads a size regression file}
\usage{
read_size_regression(filename, exceptions, repeat_length_by_marker)
}
\arguments{
\item{filename}{Path to file (character).}

\item{exceptions}{Optionally a list providing sizes for alleles not covered by the regression. See examples for how this can be used to assign sizes to X and Y at the Amelogenin locus.}

\item{repeat_length_by_marker}{Optionally a named integer vector with repeat lengths by marker. If not provided, then a .3 allele will not convert to e.g. .75 for a tetranucleotide.}
}
\value{
A function that takes a locus name and allele as arguments and returns the size.
}
\description{
Reads a size regression file
}
\details{
Read a regression file from disk and returns a function that provides the fragment length (bp) for a given locus and allele.

DNA profiles consist of the observed peaks (alleles or stutter products) at several loci as well as the peak heights and sizes. The size refers to the fragment length (bp). A linear relationship exists between the size of a peak and the size. When peaks are sampled in the \link{sample_mixture_from_genotypes} function, a size is assigned using a size regression. The \code{read_size_regression} function reads such a regression from disk.
}
\examples{
filename <- system.file("extdata",
                        "GlobalFiler_SizeRegression.csv",
                        package = "simDNAmixtures")

regression <- read_size_regression(filename)

# obtain size for the 12 allele at the vWA locus
regression("vWA", 12)

# now add AMEL sizes
regression_with_AMEL <- read_size_regression(filename, exceptions = list(
                          AMEL = stats::setNames(c(98.5, 104.5),
                                                 nm = c("X", "Y"))))
# check that we can obtain size for X at AMEL
stopifnot(regression_with_AMEL("AMEL", "X") == 98.5)

# pass in repeat_length_by_marker for more precise estimates
gf <- gf_configuration()

regression_with_repeat_length <- read_size_regression(filename,
           repeat_length_by_marker = gf$repeat_length_by_marker)

# obtain size for the 15.3 allele at the D1S1656 locus
stopifnot(regression_with_repeat_length("D1S1656", 15.3) ==
           121.628203912362 + 15.75 * 4.2170043570021)

}
