% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/api-grid.R
\name{snic_grid}
\alias{snic_grid}
\alias{snic_count_seeds}
\title{Spatial grid seeding for SNIC segmentation}
\usage{
snic_grid(
  x,
  type = c("rectangular", "diamond", "hexagonal", "random"),
  spacing,
  padding = spacing/2,
  ...
)

snic_count_seeds(x, spacing, padding = spacing/2)
}
\arguments{
\item{x}{Image data. For arrays, this must be numeric with dimensions
\code{(height, width, bands)}. For \code{SpatRaster} objects, raster
dimensions are inferred automatically.}

\item{type}{Character string indicating the spatial pattern to generate.
One of \code{"rectangular"}, \code{"diamond"}, \code{"hexagonal"},
or \code{"random"}.}

\item{spacing}{Numeric or integer. Either one value (applied to both axes)
or two values \code{(vertical, horizontal)} giving the spacing between
seeds in pixels.}

\item{padding}{Numeric or integer. Distance from image borders within which
no seeds are placed. May be of length 1 or 2. Defaults to
\code{spacing / 2}.}

\item{...}{Currently unused; reserved for future extensions.}
}
\value{
A data frame containing:
\itemize{
  \item \code{r}, \code{c} when \code{x} has no CRS.
  \item \code{lat}, \code{lon} when \code{x} has a CRS, expressed in
    \code{EPSG:4326}.
}
}
\description{
Generate seed locations on an image following one of four spatial
arrangements used in SNIC (Simple Non-Iterative Clustering) segmentation:
rectangular, diamond, hexagonal, or random. Works for both numeric arrays
and \code{\link[terra:SpatRaster-class]{SpatRaster}} objects.
}
\details{
The \code{spacing} parameter controls seed density. Padding shifts the
seed grid inward so that seeds are not placed directly on image borders.

The spatial arrangements are:
\itemize{
  \item \code{rectangular}: regular grid aligned with rows and columns.
  \item \code{diamond}: alternating row offsets, forming a diamond layout.
  \item \code{hexagonal}: alternating offsets approximating a hexagonal
    tiling.
  \item \code{random}: uniform random placement with similar expected
    density.
}

The helper \code{\link{snic_count_seeds}} estimates how many seeds would be
generated for a rectangular lattice with the given spacing and padding,
without computing coordinates. For \code{type} = \code{"diamond"} or
\code{"hexagonal"}, the actual number of seeds will be up to roughly
twice this estimate (minus boundary effects). For \code{"random"}, the
estimate corresponds to the expected density.

If \code{x} has a coordinate reference system, the returned data frame
includes geographic coordinates (\code{lat}, \code{lon}) in \code{EPSG:4326}.
}
\examples{
# Example 1: Geospatial raster
if (requireNamespace("terra", quietly = TRUE) && terra_is_working()) {
    # Load example multi-band image (Sentinel-2 subset) and downsample
    tiff_dir <- system.file("demo-geotiff",
        package = "snic",
        mustWork = TRUE
    )
    files <- file.path(tiff_dir, c(
        "S2_20LMR_B02_20220630.tif",
        "S2_20LMR_B04_20220630.tif",
        "S2_20LMR_B08_20220630.tif",
        "S2_20LMR_B12_20220630.tif"
    ))
    s2 <- terra::aggregate(terra::rast(files), fact = 8)

    # Compare grid types visually using snic_plot for immediate feedback
    types <- c("rectangular", "diamond", "hexagonal", "random")
    op <- par(mfrow = c(2, 2), mar = c(2, 2, 2, 2))
    for (tp in types) {
        seeds <- snic_grid(s2, type = tp, spacing = 12L, padding = 18L)
        snic_plot(
            s2,
            r = 4, g = 3, b = 1, stretch = "lin",
            seeds = seeds,
            main = paste("Grid:", tp)
        )
    }
    par(mfrow = c(1, 1))

    # Estimate seed counts for planning
    snic_count_seeds(s2, spacing = 12L, padding = 18L)
    par(op)
}

# Example 2: In-memory image (JPEG)
if (requireNamespace("jpeg", quietly = TRUE)) {
    img_path <- system.file(
        "demo-jpeg/clownfish.jpeg",
        package = "snic",
        mustWork = TRUE
    )
    rgb <- jpeg::readJPEG(img_path)

    # Compare grid types visually using snic_plot for immediate feedback
    types <- c("rectangular", "diamond", "hexagonal", "random")
    op <- par(mfrow = c(2, 2), mar = c(2, 2, 2, 2))
    for (tp in types) {
        seeds <- snic_grid(rgb, type = tp, spacing = 12L, padding = 18L)
        snic_plot(
            rgb,
            r = 1, g = 2, b = 3,
            seeds = seeds,
            main = paste("Grid:", tp)
        )
    }
    par(mfrow = c(1, 1))
    par(op)
}
}
\seealso{
\code{\link{snic_count_seeds}} for estimating seed counts.
}
