% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/freq.R
\name{freq}
\alias{freq}
\title{Frequency Table (spicy engine)}
\usage{
freq(
  data,
  x = NULL,
  weights = NULL,
  digits = 1,
  valid = TRUE,
  cum = FALSE,
  sort = "",
  na_val = NULL,
  labelled_levels = c("prefixed", "labels", "values", "p", "l", "v"),
  rescale = TRUE,
  styled = TRUE,
  ...
)
}
\arguments{
\item{data}{A \code{data.frame}, vector, or factor. If a data frame is provided,
specify the target variable \code{x}.}

\item{x}{A variable from \code{data} (unquoted).}

\item{weights}{Optional numeric vector of weights (same length as \code{x}).
The variable may be referenced as a bare name when it belongs to \code{data}.}

\item{digits}{Number of decimal digits to display for percentages (default: \code{1}).}

\item{valid}{Logical. If \code{TRUE} (default), display valid percentages
(excluding missing values).}

\item{cum}{Logical. If \code{TRUE}, add cumulative percentages.}

\item{sort}{Sorting method for values:
\itemize{
\item \code{""} - no sorting (default)
\item \code{"+"} - increasing frequency
\item \code{"-"} - decreasing frequency
\item \code{"name+"} - alphabetical A-Z
\item \code{"name-"} - alphabetical Z-A
}}

\item{na_val}{Vector of numeric or character values to be treated as missing (\code{NA}).

For \emph{labelled} variables (from \strong{haven} or \strong{labelled}), this argument
must refer to the underlying coded values, not the visible labels.

Example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{x <- labelled(c(1, 2, 3, 1, 2, 3), c("Low" = 1, "Medium" = 2, "High" = 3))
freq(x, na_val = 1) # Treat all "Low" as missing
}\if{html}{\out{</div>}}}

\item{labelled_levels}{For \code{labelled} variables, defines how labels and
values are displayed:
\itemize{
\item \code{"prefixed"} or \code{"p"} - show labels as \verb{[value] label} (default)
\item \code{"labels"} or \code{"l"} - show only labels
\item \code{"values"} or \code{"v"} - show only numeric codes
}}

\item{rescale}{Logical. If \code{TRUE} (default), rescale weights so that their
total equals the unweighted sample size.}

\item{styled}{Logical. If \code{TRUE} (default), print the formatted spicy table.
If \code{FALSE}, return a plain \code{data.frame} with frequency values.}

\item{...}{Additional arguments passed to \code{\link[=print.spicy_freq_table]{print.spicy_freq_table()}}.}
}
\value{
A \code{data.frame} with columns:
\itemize{
\item \code{value} - unique values or factor levels
\item \code{n} - frequency count (weighted if applicable)
\item \code{prop} - proportion of total
\item \code{valid_prop} - proportion of valid responses (if \code{valid = TRUE})
\item \code{cum_prop}, \code{cum_valid_prop} - cumulative percentages (if \code{cum = TRUE})
}

If \code{styled = TRUE}, prints the formatted table to the console and returns it invisibly.
}
\description{
Creates a frequency table for a vector or variable from a data frame, with
options for weighting, sorting, handling \emph{labelled} data, defining custom
missing values, and displaying cumulative percentages.

When \code{styled = TRUE}, the function prints a spicy-formatted ASCII table
using \code{\link[=print.spicy_freq_table]{print.spicy_freq_table()}} and \code{\link[=spicy_print_table]{spicy_print_table()}}; otherwise, it
returns a \code{data.frame} containing frequencies and proportions.
}
\details{
This function is designed to mimic common frequency procedures from
statistical software such as SPSS or Stata, while integrating the
flexibility of R's data structures.

It automatically detects the type of input (\code{vector}, \code{factor}, or
\code{labelled}) and applies appropriate transformations, including:
\itemize{
\item Handling of labelled variables via \strong{labelled} or \strong{haven}
\item Optional recoding of specific values as missing (\code{na_val})
\item Optional weighting with a rescaling mechanism
\item Support for cumulative percentages (\code{cum = TRUE})
\item Multiple display modes for labels via \code{labelled_levels}
}

When weighting is applied (\code{weights}), the frequencies and percentages are
computed proportionally to the weights. The argument \code{rescale = TRUE}
normalizes weights so their sum equals the unweighted sample size.
}
\examples{
library(labelled)

# Simple numeric vector
x <- c(1, 2, 2, 3, 3, 3, NA)
freq(x)

# Labelled variable (haven-style)
x_lbl <- labelled(
  c(1, 2, 3, 1, 2, 3, 1, 2, NA),
  labels = c("Low" = 1, "Medium" = 2, "High" = 3)
)
var_label(x_lbl) <- "Satisfaction level"

# Treat value 1 ("Low") as missing
freq(x_lbl, na_val = 1)

# Display only labels, add cumulative \%
freq(x_lbl, labelled_levels = "labels", cum = TRUE)

# Display values only, sorted descending
freq(x_lbl, labelled_levels = "values", sort = "-")

# With weighting
df <- data.frame(
  sexe = factor(c("Male", "Female", "Female", "Male", NA, "Female")),
  poids = c(12, 8, 10, 15, 7, 9)
)

# Weighted frequencies (normalized)
freq(df, sexe, weights = poids, rescale = TRUE)

# Weighted frequencies (without rescaling)
freq(df, sexe, weights = poids, rescale = FALSE)

# Base R style, with weights and cumulative percentages
freq(df$sexe, weights = df$poids, cum = TRUE)

# Piped version (tidy syntax) and sort alphabetically descending ("name-")
df |> freq(sexe, sort = "name-")

# Non-styled return (for programmatic use)
f <- freq(df, sexe, styled = FALSE)
head(f)

}
\seealso{
\code{\link[=print.spicy_freq_table]{print.spicy_freq_table()}} for formatted printing.
\code{\link[=spicy_print_table]{spicy_print_table()}} for the underlying ASCII rendering engine.
}
