% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixture_mixed-data.R
\name{mx_mixed_lca}
\alias{mx_mixed_lca}
\title{Estimate mixed data latent class analysis using OpenMx}
\usage{
mx_mixed_lca(
  data = NULL,
  classes = 1L,
  variances = "equal",
  covariances = "zero",
  run = TRUE,
  expand_grid = FALSE,
  ...
)
}
\arguments{
\item{data}{The data.frame to be used for model fitting.}

\item{classes}{A vector of integers, indicating which class solutions to
generate. Defaults to 1L. E.g., \code{classes = 1:6},}

\item{variances}{Character vector. Specifies which variance components to
estimate. Defaults to "equal" (constrain variances across classes); the
other option is "varying" (estimate variances freely across classes). Each
element of this vector refers to one of the models you wish to run.}

\item{covariances}{Character vector. Specifies which covariance components to
estimate. Defaults to "zero" (covariances constrained to zero; this
corresponds
to an assumption of conditional independence of the indicators); other
options are "equal" (covariances between items constrained to be equal across
classes), and "varying" (free covariances across classes).}

\item{run}{Logical, whether or not to run the model. If \code{run = TRUE},
the function calls \code{\link{mixture_starts}} and \code{\link{run_mx}}.}

\item{expand_grid}{Logical, whether or not to estimate all possible combinations of the \code{variances} and \code{covariances} arguments. Defaults to \code{FALSE}.}

\item{...}{Additional arguments, passed to functions.}
}
\value{
A list of class \code{mixture_list}.
}
\description{
This function simplifies the specification of latent class models with mixed
data types:
models that estimate membership of a categorical latent variable based on
binary/ordinal and continuous indicators. See Details for more information.
}
\details{
The procedure is as follows:
\enumerate{
\item Construct a latent profile model for the continuous indicators using
\code{\link[tidySEM:mx_profiles]{mx_profiles()}}.
\item Construct a latent class model for the categorical indicators using
\code{\link[tidySEM:mx_lca]{mx_lca()}}.
\item Combine the models from steps 1. and 2. into one joint model.
}

If \code{run = TRUE}, simulated annealing is used to estimate the mixture model,
as explained in Van Lissa, Garnier-Villareal, & Anadria (2023). However, the
inclusion of categorical indicators often leads to a large ordinal error,
which automatically initiates a final optimization step using
\code{\link[OpenMx:mxTryHardOrdinal]{OpenMx::mxTryHardOrdinal()}}.
}
\examples{
\dontrun{
if(isTRUE(requireNamespace("OpenMx", quietly = TRUE))) {
library(tidySEM)
library(OpenMx)
# Construct dataset with ordinal and categorical indicators
set.seed(1)
n = 200
mns <- c(rep(0, floor(.3*n)), rep(2, ceiling(.7*n)))
df <- rnorm(4*n, mean = rep(mns, 4))
df <- matrix(df, nrow = n)
df <- t(t(df) * c(1, 2, .5, 1))
df <- data.frame(df)
df$X4 <- cut(df$X4, 3, labels = FALSE)
df$X4 <- OpenMx::mxFactor(df$X4, levels = c(1:3))
# Estimate the model
set.seed(1)
res <- mx_mixed_lca(data = df, classes = 2)
}
}
}
\references{
Van Lissa, C. J., Garnier-Villarreal, M., & Anadria, D. (2023).
Recommended Practices in Latent Class Analysis using the Open-Source
R-Package tidySEM. Structural Equation Modeling.
\doi{10.1080/10705511.2023.2250920}
}
\keyword{mixture}
\keyword{models}
\keyword{openmx}
