# Simple test vectors
x_na <- c(1, NA, 3)
x_clean <- c(1, 2, 3)
x_nan <- c(1, NaN, 5)
x_na_and_nan <- c(1, NA, NaN, 5)
x_inf <- c(1, Inf, 3)
x_neginf <- c(-Inf, 2, 3)
x_both_inf <- c(-Inf, 0, Inf)

# min ----
test_that("min removes NA and warns", {
  expect_warning(result <- min(x_na), "missing value")
  expect_equal(result, 1)
})

test_that("min with no NAs produces no warning", {
  expect_no_warning(result <- min(x_clean))
  expect_equal(result, 1)
})

test_that("min with na.rm = FALSE returns NA", {
  expect_no_warning(result <- min(x_na, na.rm = FALSE))
  expect_true(is.na(result))
})

test_that("min handles NaN with separate warning", {
  expect_warning(result <- min(x_nan), "NaN value")
  expect_equal(result, 1)
})

test_that("min with both NA and NaN issues both warnings", {
  warnings <- testthat::capture_warnings(result <- min(x_na_and_nan))
  expect_equal(result, 1)
  expect_length(warnings, 2)
  expect_true(any(grepl("missing value", warnings)))
  expect_true(any(grepl("NaN value", warnings)))
})

test_that("min handles Inf", {
  expect_no_warning(result <- min(x_inf))
  expect_equal(result, 1)
})

test_that("min handles -Inf", {
  expect_no_warning(result <- min(x_neginf))
  expect_equal(result, -Inf)
})

test_that("min of mixed Inf returns -Inf", {
  expect_no_warning(result <- min(x_both_inf))
  expect_equal(result, -Inf)
})

# max ----
test_that("max removes NA and warns", {
  expect_warning(result <- max(x_na), "missing value")
  expect_equal(result, 3)
})

test_that("max with no NAs produces no warning", {
  expect_no_warning(result <- max(x_clean))
  expect_equal(result, 3)
})

test_that("max handles NaN with separate warning", {
  expect_warning(result <- max(x_nan), "NaN value")
  expect_equal(result, 5)
})

test_that("max with both NA and NaN issues both warnings", {
  warnings <- testthat::capture_warnings(result <- max(x_na_and_nan))
  expect_equal(result, 5)
  expect_length(warnings, 2)
  expect_true(any(grepl("missing value", warnings)))
  expect_true(any(grepl("NaN value", warnings)))
})

test_that("max handles Inf", {
  expect_no_warning(result <- max(x_inf))
  expect_equal(result, Inf)
})

test_that("max handles -Inf", {
  expect_no_warning(result <- max(x_neginf))
  expect_equal(result, 3)
})

test_that("max of mixed Inf returns Inf", {
  expect_no_warning(result <- max(x_both_inf))
  expect_equal(result, Inf)
})

# Edge cases: all-NA throws error ----
test_that("min of all-NA throws error", {
  expect_error(min(c(NA, NA)), "check if something went wrong")
})

test_that("max of all-NA throws error", {
  expect_error(max(c(NA, NA)), "check if something went wrong")
})

test_that("min of all-NaN throws error", {
  expect_error(min(c(NaN, NaN)), "check if something went wrong")
})

test_that("max of all-NaN throws error", {
  expect_error(max(c(NaN, NaN)), "check if something went wrong")
})

# Edge cases: empty vector preserves base R behavior ----
test_that("min of empty vector returns Inf with base R warning", {
  # Base R emits warning about no non-missing arguments
  expect_warning(result <- min(numeric(0)), "no non-missing arguments")
  expect_equal(result, Inf)
})

test_that("max of empty vector returns -Inf with base R warning", {
  # Base R emits warning about no non-missing arguments
  expect_warning(result <- max(numeric(0)), "no non-missing arguments")
  expect_equal(result, -Inf)
})

# Multiple arguments ----
test_that("min with multiple vectors containing NA", {
  expect_warning(result <- min(c(5, NA), c(1, 2)), "missing value")
  expect_equal(result, 1)
})

test_that("max with multiple vectors containing NA", {
  expect_warning(result <- max(c(1, NA), c(5, 2)), "missing value")
  expect_equal(result, 5)
})

# range ----
test_that("range removes NA and warns", {
  expect_warning(result <- range(x_na), "missing value")
  expect_equal(result, c(1, 3))
})

test_that("range with no NAs produces no warning", {
  expect_no_warning(result <- range(x_clean))
  expect_equal(result, c(1, 3))
})

test_that("range with na.rm = FALSE returns NA", {
  expect_no_warning(result <- range(x_na, na.rm = FALSE))
  expect_true(all(is.na(result)))
})

test_that("range handles NaN with separate warning", {
  expect_warning(result <- range(x_nan), "NaN value")
  expect_equal(result, c(1, 5))
})

test_that("range with both NA and NaN issues both warnings", {
  warnings <- testthat::capture_warnings(result <- range(x_na_and_nan))
  expect_equal(result, c(1, 5))
  expect_length(warnings, 2)
  expect_true(any(grepl("missing value", warnings)))
  expect_true(any(grepl("NaN value", warnings)))
})

test_that("range handles Inf without finite", {
  expect_no_warning(result <- range(x_inf))
  expect_equal(result, c(1, Inf))
})

test_that("range handles -Inf without finite", {
  expect_no_warning(result <- range(x_neginf))
  expect_equal(result, c(-Inf, 3))
})

test_that("range of mixed Inf returns both", {
  expect_no_warning(result <- range(x_both_inf))
  expect_equal(result, c(-Inf, Inf))
})

test_that("range with finite = TRUE removes Inf and warns", {
  expect_warning(result <- range(x_inf, finite = TRUE), "infinite value")
  expect_equal(result, c(1, 3))
})

test_that("range with finite = TRUE removes -Inf and warns", {
  expect_warning(result <- range(x_neginf, finite = TRUE), "infinite value")
  expect_equal(result, c(2, 3))
})

test_that("range with finite = TRUE removes both Inf/-Inf", {
  expect_warning(result <- range(x_both_inf, finite = TRUE), "infinite value")
  expect_equal(result, c(0, 0))
})

test_that("range with finite = TRUE removes NA and Inf with both warnings", {
  x <- c(1, NA, Inf, 5)
  warnings <- testthat::capture_warnings(result <- range(x, finite = TRUE))
  expect_equal(result, c(1, 5))
  expect_length(warnings, 2)
  expect_true(any(grepl("missing value", warnings)))
  expect_true(any(grepl("infinite value", warnings)))
})

test_that("range of all-NA throws error", {
  expect_error(range(c(NA, NA)), "check if something went wrong")
})

test_that("range of all-NaN throws error", {
  expect_error(range(c(NaN, NaN)), "check if something went wrong")
})

test_that("range of all non-finite throws error when finite = TRUE", {
  expect_error(range(c(Inf, -Inf, NA), finite = TRUE), "check if something went wrong")
})

test_that("range of empty vector returns Inf/-Inf with base R warning", {
  expect_warning(result <- range(numeric(0)), "no non-missing arguments")
  expect_equal(result, c(Inf, -Inf))
})

test_that("range with multiple vectors containing NA", {
  expect_warning(result <- range(c(5, NA), c(1, 2)), "missing value")
  expect_equal(result, c(1, 5))
})

# pmax ----
test_that("pmax removes NA and warns", {
  expect_warning(
    result <- pmax(c(1, NA, 3), c(2, 2, NA)),
    "missing value"
  )
  expect_equal(result, c(2, 2, 3))
})

test_that("pmax warns about all-NA positions", {
  expect_warning(
    result <- pmax(c(NA, 1), c(NA, 2)),
    "all NA"
  )
  expect_equal(result, c(NA, 2))
})

test_that("pmax warns separately about removed NAs and all-NA positions", {
  # c(NA, NA, 1) and c(NA, 2, NA): position 1 is all-NA, positions 2 and 3 have removable NAs
  warnings <- testthat::capture_warnings(
    result <- pmax(c(NA, NA, 1), c(NA, 2, NA))
  )
  expect_equal(result, c(NA, 2, 1))
  expect_length(warnings, 2)
  expect_true(any(grepl("all NA", warnings)))
  expect_true(any(grepl("missing value", warnings)))
})

test_that("pmax with na.rm = FALSE returns NA", {
  expect_no_warning(result <- pmax(c(1, NA), c(2, 2), na.rm = FALSE))
  expect_equal(result, c(2, NA))
})

test_that("pmax with no NAs produces no warning", {
  expect_no_warning(result <- pmax(c(1, 2), c(2, 1)))
  expect_equal(result, c(2, 2))
})

test_that("pmax preserves attributes from first argument", {
  x <- c(a = 1, b = 2)
  y <- c(2, 1)
  result <- pmax(x, y)
  expect_equal(names(result), c("a", "b"))
})

test_that("pmax errors when all positions are all-NA", {
  expect_error(
    pmax(c(NA, NA), c(NA, NA)),
    "All values are NA"
  )
})

test_that("pmax handles recycling", {
  expect_no_warning(result <- pmax(c(1, 2, 3), 2))
  expect_equal(result, c(2, 2, 3))
})

test_that("pmax with multiple arguments", {
  expect_no_warning(result <- pmax(c(1, 5), c(3, 2), c(2, 4)))
  expect_equal(result, c(3, 5))
})

# pmin ----
test_that("pmin removes NA and warns", {
  expect_warning(
    result <- pmin(c(1, NA, 3), c(2, 2, NA)),
    "missing value"
  )
  expect_equal(result, c(1, 2, 3))
})

test_that("pmin warns about all-NA positions", {
  expect_warning(
    result <- pmin(c(NA, 1), c(NA, 2)),
    "all NA"
  )
  expect_equal(result, c(NA, 1))
})

test_that("pmin warns separately about removed NAs and all-NA positions", {
  # c(NA, NA, 3) and c(NA, 2, NA): position 1 is all-NA, positions 2 and 3 have removable NAs
  warnings <- testthat::capture_warnings(
    result <- pmin(c(NA, NA, 3), c(NA, 2, NA))
  )
  expect_equal(result, c(NA, 2, 3))
  expect_length(warnings, 2)
  expect_true(any(grepl("all NA", warnings)))
  expect_true(any(grepl("missing value", warnings)))
})

test_that("pmin with na.rm = FALSE returns NA", {
  expect_no_warning(result <- pmin(c(1, NA), c(2, 2), na.rm = FALSE))
  expect_equal(result, c(1, NA))
})

test_that("pmin with no NAs produces no warning", {
  expect_no_warning(result <- pmin(c(1, 2), c(2, 1)))
  expect_equal(result, c(1, 1))
})

test_that("pmin errors when all positions are all-NA", {
  expect_error(
    pmin(c(NA, NA), c(NA, NA)),
    "All values are NA"
  )
})

test_that("pmin handles recycling", {
  expect_no_warning(result <- pmin(c(1, 2, 3), 2))
  expect_equal(result, c(1, 2, 2))
})

test_that("pmin with multiple arguments", {
  expect_no_warning(result <- pmin(c(1, 5), c(3, 2), c(2, 4)))
  expect_equal(result, c(1, 2))
})
