#' @title Frequency Plot of DBE - O
#' @name uplot_dbe_minus_o_freq
#' @family plots
#'
#' @description
#' Creates a bar plot showing the frequency distribution of `dbe_o`
#' (DBE minus oxygen). The plot uses the unified UME plotting theme and
#' optionally adds a small UME caption. A Plotly version can be returned.
#'
#' @inheritParams main_docu
#'
#' @param df A data.table containing at least the column `dbe_o`.
#'
#' @return A ggplot2 object or, if requested, a Plotly object.
#'
#' @import data.table
#' @import ggplot2
#' @importFrom plotly ggplotly
#'
#' @examples
#' uplot_dbe_minus_o_freq(mf_data_demo)
#'
#' @export
uplot_dbe_minus_o_freq <- function(
    df,
    gg_size = 12,
    logo = TRUE,
    plotly = FALSE,
    ...
) {

  # --- Validation ---------------------------------------------------------------
  if (!"dbe_o" %in% names(df)) {
    stop("Column 'dbe_o' not found in provided data.")
  }

  if (nrow(df) == 0) {
    stop("The provided data frame is empty.")
  }

  df <- data.table::as.data.table(df)

  # --- Frequency table ----------------------------------------------------------
  tab <- df[, .N, by = .(dbe_o)]
  data.table::setorder(tab, dbe_o)

  # --- Base ggplot --------------------------------------------------------------
  p <- ggplot(tab, aes(x = factor(dbe_o), y = N)) +
    geom_col(fill = "grey70", color = "black", width = 0.8) +
    labs(
      title = "DBE - O Frequency",
      x = "DBE - O",
      y = "Count",
      caption = if (logo) "UltraMassExplorer" else NULL
    ) +
    theme_uplots(base_size = gg_size)

  # --- Return Plotly version if requested ---------------------------------------
  if (isTRUE(plotly)) {

    pp <- plotly::ggplotly(p)

    # Add caption explicitly in Plotly (ggplotly drops captions)
    if (logo) {
      pp <- pp |>
        plotly::layout(
          annotations = list(
            list(
              text = "<i>UltraMassExplorer</i>",
              xref = "paper", yref = "paper",
              x = 1, y = -0.12,
              xanchor = "right", yanchor = "top",
              showarrow = FALSE,
              font = list(size = gg_size, color = "gray40")
            )
          )
        )
    }

    return(pp)
  }

  return(p)
}
