% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cusum.R
\name{cusum_failure}
\alias{cusum_failure}
\title{Cumulative Sum of Failures}
\usage{
cusum_failure(xi, p0, p1, by = NULL, alpha = 0.05, beta = 0.05)
}
\arguments{
\item{xi}{An integer. The dichotomous outcome variable (1 = Failure, 0 = Success)
for the i-th procedure.}

\item{p0}{A double. The acceptable event rate.}

\item{p1}{A double. The unacceptable event rate.}

\item{by}{A factor. Optional variable to stratify procedures by.}

\item{alpha}{A double. The Type I Error rate. Probability of rejecting the
null hypothesis when `p0` is the true rate.}

\item{beta}{A double. The Type II Error rate. Probability of failing to reject
null hypothesis when it is false.}
}
\value{
An object of class \code{data.frame}.
}
\description{
Calculates the cumulative sum of failures for a series of
procedures which can be used to create CUSUM charts.
}
\examples{
library(purrr)
library(ggplot2)

# Data
df <- data.frame(
  xi = simplify(
    map(
      c(.1,.08,.05,.1,.13,.14,.14,.09,.25),
      ~ rbinom(50,1,.x))),
   p0 = simplify(
    map(
      c(.1,.1,.1,.1,.1,.1,.1,.15,.2),
      ~ rnorm(50,.x,.03))),
   by = rep(
    factor(paste('Subject', c('A','B','C'))),
    times = c(150,150,150))
 )

# Overall event rate
p0 <- sum(df$xi) / nrow(df)

# Create CUSUM plot
cusum_failure(
 xi = df$xi,
 p0 = p0,
 p1 = p0 * 1.5,
 by = df$by
) |>
ggplot(aes(y = cusum, x = i)) +
  geom_step() +
  geom_line(mapping = aes(y = l0), linetype = 2) +
  geom_line(mapping = aes(y = l1), linetype = 2) +
  ylab("Cumulative Failures") +
  xlab("Case Number") +
  facet_wrap(~ by) +
  theme_bw()
}
\references{
Rogers, C. A., Reeves, B. C., Caputo, M., Ganesh, J. S., Bonser, R. S., & Angelini, G. D. (2004). Control chart methods for monitoring cardiac surgical performance and their interpretation. The Journal of Thoracic and Cardiovascular Surgery, 128(6), 811-819.
}
