#' @title Model Comparison and Summary Functions
#' @name summary_comparison
#' @description Functions for comparing models and generating summary tables.
NULL

#' Compare All Models
#'
#' Generates a comprehensive comparison table of all fitted models.
#'
#' @param twoway_result Result from fit_twoway_fe (or NULL).
#' @param mundlak_result Result from fit_mundlak_cre (or NULL).
#' @param bayes_result Result from fit_bayesian_hierarchical (or NULL).
#' @param pls_result Result from fit_pls_multivariate (or NULL).
#'
#' @return A data frame with model comparison metrics.
#'
#' @examples
#' \donttest{
#' if (requireNamespace("plm", quietly = TRUE)) {
#'   set.seed(123)
#'   panel <- data.frame(
#'     year = rep(2000:2019, 5),
#'     sector = rep(LETTERS[1:5], each = 20),
#'     log_direct = rnorm(100, 5, 0.5),
#'     log_production = rnorm(100, 5, 0.5)
#'   )
#'   panel$log_production <- panel$log_direct * 0.95 + rnorm(100, 0, 0.1)
#'
#'   twoway <- fit_twoway_fe(panel)
#'   mundlak <- fit_mundlak_cre(panel)
#'
#'   comparison <- compare_models(
#'     twoway_result = twoway,
#'     mundlak_result = mundlak
#'   )
#'   print(comparison)
#' }
#' }
#'
#' @export
compare_models <- function(twoway_result = NULL,
                            mundlak_result = NULL,
                            bayes_result = NULL,
                            pls_result = NULL) {

    rows <- list()

    if (!is.null(twoway_result)) {
        r2 <- twoway_result$r2_within
        metrics <- twoway_result$metrics
        rows$twoway <- data.frame(
            model = "Two-Way FE",
            R2 = if (is.finite(r2)) round(r2, 4) else NA,
            MAE_log = if (!is.null(metrics)) round(metrics$mae_log, 4) else NA,
            RMSE_log = if (!is.null(metrics)) round(metrics$rmse_log, 4) else NA,
            MAE_rel_range = if (!is.null(metrics)) round(metrics$mae_rel_range, 2) else NA,
            stringsAsFactors = FALSE
        )
    }

    if (!is.null(mundlak_result)) {
        metrics <- mundlak_result$metrics
        r2 <- compute_r2(
            mundlak_result$panel_data_augmented$log_production,
            as.numeric(stats::predict(mundlak_result$model))
        )
        rows$mundlak <- data.frame(
            model = "Mundlak CRE",
            R2 = if (is.finite(r2)) round(r2, 4) else NA,
            MAE_log = if (!is.null(metrics)) round(metrics$mae_log, 4) else NA,
            RMSE_log = if (!is.null(metrics)) round(metrics$rmse_log, 4) else NA,
            MAE_rel_range = if (!is.null(metrics)) round(metrics$mae_rel_range, 2) else NA,
            stringsAsFactors = FALSE
        )
    }

    if (!is.null(bayes_result)) {
        r2 <- bayes_result$r2_bayes
        metrics <- bayes_result$metrics
        rows$bayes <- data.frame(
            model = "Bayesian Hierarchical",
            R2 = if (is.finite(r2)) round(r2, 4) else NA,
            MAE_log = if (!is.null(metrics)) round(metrics$mae_log, 4) else NA,
            RMSE_log = if (!is.null(metrics)) round(metrics$rmse_log, 4) else NA,
            MAE_rel_range = if (!is.null(metrics)) round(metrics$mae_rel_range, 2) else NA,
            stringsAsFactors = FALSE
        )
    }

    if (!is.null(pls_result)) {
        r2 <- pls_result$metrics_cv$R2_CV
        metrics <- pls_result$metrics_insample
        rows$pls <- data.frame(
            model = sprintf("PLS (%d comp)", pls_result$optimal_ncomp),
            R2 = if (is.finite(r2)) round(r2, 4) else NA,
            MAE_log = if (!is.null(metrics)) round(metrics$mae_log, 4) else NA,
            RMSE_log = if (!is.null(metrics)) round(metrics$rmse_log, 4) else NA,
            MAE_rel_range = if (!is.null(metrics)) round(metrics$mae_rel_range, 2) else NA,
            stringsAsFactors = FALSE
        )
    }

    if (length(rows) == 0L) {
        warning("No models provided for comparison.")
        return(data.frame(
            model = character(0),
            R2 = numeric(0),
            MAE_log = numeric(0),
            RMSE_log = numeric(0),
            MAE_rel_range = numeric(0)
        ))
    }

    do.call(rbind, rows)
}


#' Generate Comprehensive Analysis Summary
#'
#' Creates a complete summary of all analysis results including models,
#' validation, and structural break tests.
#'
#' @param comparison_table Data frame from compare_models.
#' @param cv_summary Data frame from summarize_cv_results (or NULL).
#' @param break_results Result from test_structural_breaks (or NULL).
#' @param cca_results Result from run_sparse_cca (or NULL).
#' @param granger_results Data frame from panel_granger_test (or NULL).
#'
#' @return A list with formatted summary components.
#'
#' @examples
#' comparison <- data.frame(
#'   model = c("Model A", "Model B"),
#'   R2 = c(0.95, 0.92),
#'   RMSE_log = c(0.05, 0.06)
#' )
#' summary_list <- generate_analysis_summary(comparison)
#' print(summary_list$best_model)
#'
#' @export
generate_analysis_summary <- function(comparison_table,
                                       cv_summary = NULL,
                                       break_results = NULL,
                                       cca_results = NULL,
                                       granger_results = NULL) {

    summary_list <- list()

    summary_list$comparison_table <- comparison_table

    if (nrow(comparison_table) > 0L && "R2" %in% names(comparison_table)) {
        valid_r2 <- comparison_table[is.finite(comparison_table$R2), ]
        if (nrow(valid_r2) > 0L) {
            best_idx <- which.max(valid_r2$R2)
            summary_list$best_model_r2 <- list(
                model = valid_r2$model[best_idx],
                R2 = valid_r2$R2[best_idx]
            )
        }
    }

    if (nrow(comparison_table) > 0L && "RMSE_log" %in% names(comparison_table)) {
        valid_rmse <- comparison_table[is.finite(comparison_table$RMSE_log), ]
        if (nrow(valid_rmse) > 0L) {
            best_idx <- which.min(valid_rmse$RMSE_log)
            summary_list$best_model_rmse <- list(
                model = valid_rmse$model[best_idx],
                RMSE_log = valid_rmse$RMSE_log[best_idx]
            )
        }
    }

    if (!is.null(cv_summary) && nrow(cv_summary) > 0L) {
        summary_list$cv_summary <- cv_summary

        all_partition <- cv_summary[cv_summary$partition == "all", ]
        if (nrow(all_partition) > 0L) {
            best_oos_idx <- which.min(all_partition$rmse_mean)
            summary_list$best_oos_model <- list(
                model = all_partition$model[best_oos_idx],
                rmse_mean = all_partition$rmse_mean[best_oos_idx]
            )
        }
    }

    if (!is.null(break_results)) {
        summary_list$breaks <- list(
            cusum = break_results$cusum,
            mosum = break_results$mosum,
            supf = break_results$supf,
            breakpoints = break_results$breakpoints
        )
    }

    if (!is.null(cca_results)) {
        summary_list$cca <- list(
            r1 = cca_results$correlations[1L],
            sum_r_squared = cca_results$sum_r_squared,
            n_pcs = c(x = cca_results$n_pcs_x, y = cca_results$n_pcs_y)
        )
    }

    if (!is.null(granger_results) && nrow(granger_results) > 0L) {
        summary_list$granger <- granger_results
    }

    summary_list
}


#' Print Analysis Summary
#'
#' Prints a formatted summary of analysis results to console.
#'
#' @param summary_list Result from generate_analysis_summary.
#' @param verbose Logical. Print detailed output. Default TRUE.
#'
#' @return Invisible NULL.
#'
#' @examples
#' comparison <- data.frame(
#'   model = c("Two-Way FE", "Mundlak CRE"),
#'   R2 = c(0.95, 0.92),
#'   RMSE_log = c(0.05, 0.06)
#' )
#' summary_list <- generate_analysis_summary(comparison)
#' print_analysis_summary(summary_list)
#'
#' @export
print_analysis_summary <- function(summary_list, verbose = TRUE) {

    cat("\n")
    cat("========================================\n")
    cat("VALUE-PRICE ANALYSIS SUMMARY\n")
    cat("========================================\n\n")

    if (!is.null(summary_list$comparison_table)) {
        cat("--- Model Comparison ---\n")
        print(summary_list$comparison_table, row.names = FALSE)
        cat("\n")
    }

    if (!is.null(summary_list$best_model_r2)) {
        cat(sprintf("Best R-squared: %s (R2 = %.4f)\n",
                    summary_list$best_model_r2$model,
                    summary_list$best_model_r2$R2))
    }

    if (!is.null(summary_list$best_model_rmse)) {
        cat(sprintf("Best RMSE (in-sample): %s (RMSE = %.4f)\n",
                    summary_list$best_model_rmse$model,
                    summary_list$best_model_rmse$RMSE_log))
    }

    if (!is.null(summary_list$best_oos_model)) {
        cat(sprintf("Best OOS: %s (RMSE = %.4f)\n",
                    summary_list$best_oos_model$model,
                    summary_list$best_oos_model$rmse_mean))
    }

    if (verbose && !is.null(summary_list$cv_summary)) {
        cat("\n--- Out-of-Sample Validation ---\n")
        print(summary_list$cv_summary, row.names = FALSE)
    }

    if (!is.null(summary_list$cca)) {
        cat("\n--- CCA Results ---\n")
        cat(sprintf("First canonical correlation: %.4f\n", summary_list$cca$r1))
        cat(sprintf("Sum of squared correlations: %.4f\n", summary_list$cca$sum_r_squared))
    }

    if (!is.null(summary_list$granger)) {
        cat("\n--- Granger Causality ---\n")
        print(summary_list$granger, row.names = FALSE)
    }

    if (!is.null(summary_list$breaks)) {
        cat("\n--- Structural Break Tests ---\n")

        if (!is.null(summary_list$breaks$cusum) &&
            summary_list$breaks$cusum$available) {
            cat(sprintf("CUSUM: stat = %.3f, p = %.4f\n",
                        summary_list$breaks$cusum$statistic,
                        summary_list$breaks$cusum$p_value))
        }

        if (!is.null(summary_list$breaks$mosum) &&
            summary_list$breaks$mosum$available) {
            cat(sprintf("MOSUM: stat = %.3f, p = %.4f\n",
                        summary_list$breaks$mosum$statistic,
                        summary_list$breaks$mosum$p_value))
        }

        if (!is.null(summary_list$breaks$supf) &&
            summary_list$breaks$supf$available) {
            cat(sprintf("supF: stat = %.3f, p = %.4f\n",
                        summary_list$breaks$supf$statistic,
                        summary_list$breaks$supf$p_value))
        }

        if (!is.null(summary_list$breaks$breakpoints)) {
            cat(sprintf("Estimated breakpoints: %s\n",
                        paste(summary_list$breaks$breakpoints, collapse = ", ")))
        }
    }

    cat("\n========================================\n")

    invisible(NULL)
}


#' Export Results to CSV
#'
#' Exports analysis results to CSV files.
#'
#' @param comparison_table Data frame from compare_models.
#' @param cv_results Data frame from rolling_window_cv.
#' @param sector_summary Data frame from fit_bayesian_glm_sectors.
#' @param output_dir Directory for output files. Default tempdir().
#' @param prefix Filename prefix. Default "valueprhr".
#'
#' @return Character vector of created file paths.
#'
#' @examples
#' comparison <- data.frame(
#'   model = c("Model A", "Model B"),
#'   R2 = c(0.95, 0.92)
#' )
#' files <- export_results_csv(comparison_table = comparison)
#' print(files)
#'
#' @export
export_results_csv <- function(comparison_table = NULL,
                                cv_results = NULL,
                                sector_summary = NULL,
                                output_dir = tempdir(),
                                prefix = "valueprhr") {

    if (!dir.exists(output_dir)) {
        dir.create(output_dir, recursive = TRUE)
    }

    files_created <- character(0)

    if (!is.null(comparison_table)) {
        file_path <- file.path(output_dir, paste0(prefix, "_comparison.csv"))
        utils::write.csv(comparison_table, file_path, row.names = FALSE)
        files_created <- c(files_created, file_path)
    }

    if (!is.null(cv_results)) {
        file_path <- file.path(output_dir, paste0(prefix, "_cv_results.csv"))
        utils::write.csv(cv_results, file_path, row.names = FALSE)
        files_created <- c(files_created, file_path)
    }

    if (!is.null(sector_summary)) {
        file_path <- file.path(output_dir, paste0(prefix, "_sector_summary.csv"))
        utils::write.csv(sector_summary, file_path, row.names = FALSE)
        files_created <- c(files_created, file_path)
    }

    if (length(files_created) > 0L) {
        message(sprintf("Created %d file(s) in %s", length(files_created), output_dir))
    }

    files_created
}


#' Run Complete Analysis Pipeline
#'
#' Convenience function to run the full analysis pipeline.
#'
#' @param direct_prices Data frame with direct prices.
#' @param production_prices Data frame with production prices.
#' @param run_bayesian Logical. Run Bayesian models. Default FALSE.
#' @param run_cv Logical. Run cross-validation. Default TRUE.
#' @param run_breaks Logical. Run structural break tests. Default TRUE.
#' @param verbose Logical. Print progress. Default TRUE.
#'
#' @return A list with all analysis results.
#'
#' @examples
#' \donttest{
#' set.seed(123)
#' years <- 2000:2019
#' sectors <- LETTERS[1:5]
#'
#' direct <- data.frame(Year = years)
#' production <- data.frame(Year = years)
#' for (s in sectors) {
#'   direct[[s]] <- 100 + cumsum(rnorm(20, 2, 1))
#'   production[[s]] <- 102 + cumsum(rnorm(20, 2, 1))
#' }
#'
#' if (requireNamespace("plm", quietly = TRUE)) {
#'   results <- run_full_analysis(
#'     direct, production,
#'     run_bayesian = FALSE,
#'     run_cv = FALSE
#'   )
#'   print(results$comparison)
#' }
#' }
#'
#' @export
run_full_analysis <- function(direct_prices,
                               production_prices,
                               run_bayesian = FALSE,
                               run_cv = TRUE,
                               run_breaks = TRUE,
                               verbose = TRUE) {

    results <- list()

    if (verbose) message("Preparing panel data...")
    panel_data <- prepare_panel_data(direct_prices, production_prices)
    results$panel_data <- panel_data

    if (verbose) message("Fitting Two-Way FE model...")
    results$twoway <- tryCatch(
        fit_twoway_fe(panel_data),
        error = function(e) {
            warning("Two-Way FE failed: ", conditionMessage(e))
            NULL
        }
    )

    if (verbose) message("Fitting Mundlak CRE model...")
    results$mundlak <- tryCatch(
        fit_mundlak_cre(panel_data),
        error = function(e) {
            warning("Mundlak CRE failed: ", conditionMessage(e))
            NULL
        }
    )

    if (run_bayesian && requireNamespace("rstanarm", quietly = TRUE)) {
        if (verbose) message("Fitting Bayesian hierarchical model...")
        results$bayes <- tryCatch(
            fit_bayesian_hierarchical(panel_data, chains = 2L, iter = 2000L),
            error = function(e) {
                warning("Bayesian model failed: ", conditionMessage(e))
                NULL
            }
        )
    }

    if (verbose) message("Fitting PLS model...")
    matrices <- prepare_log_matrices(direct_prices, production_prices)
    results$pls <- tryCatch(
        fit_pls_multivariate(matrices$X_clean, matrices$Y_clean),
        error = function(e) {
            warning("PLS failed: ", conditionMessage(e))
            NULL
        }
    )

    if (verbose) message("Creating model comparison...")
    results$comparison <- compare_models(
        twoway_result = results$twoway,
        mundlak_result = results$mundlak,
        bayes_result = if (exists("bayes", results)) results$bayes else NULL,
        pls_result = results$pls
    )

    if (run_cv) {
        if (verbose) message("Running cross-validation...")
        results$cv <- tryCatch(
            rolling_window_cv(panel_data, verbose = FALSE),
            error = function(e) {
                warning("CV failed: ", conditionMessage(e))
                NULL
            }
        )

        if (!is.null(results$cv)) {
            results$cv_summary <- summarize_cv_results(results$cv)
        }
    }

    if (run_breaks && requireNamespace("strucchange", quietly = TRUE)) {
        if (verbose) message("Testing structural breaks...")
        results$breaks <- tryCatch(
            test_structural_breaks(panel_data),
            error = function(e) {
                warning("Break tests failed: ", conditionMessage(e))
                NULL
            }
        )
    }

    results$summary <- generate_analysis_summary(
        comparison_table = results$comparison,
        cv_summary = if (exists("cv_summary", results)) results$cv_summary else NULL,
        break_results = if (exists("breaks", results)) results$breaks else NULL
    )

    if (verbose) {
        print_analysis_summary(results$summary)
    }

    results
}
