% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prob-gain_curve.R
\name{gain_curve}
\alias{gain_curve}
\alias{gain_curve.data.frame}
\title{Gain curve}
\usage{
gain_curve(data, ...)

\method{gain_curve}{data.frame}(
  data,
  truth,
  ...,
  na_rm = TRUE,
  event_level = yardstick_event_level(),
  case_weights = NULL
)
}
\arguments{
\item{data}{A \code{data.frame} containing the columns specified by \code{truth} and
\code{...}.}

\item{...}{A set of unquoted column names or one or more
\code{dplyr} selector functions to choose which variables contain the
class probabilities. If \code{truth} is binary, only 1 column should be selected,
and it should correspond to the value of \code{event_level}. Otherwise, there
should be as many columns as factor levels of \code{truth} and the ordering of
the columns should be the same as the factor levels of \code{truth}.}

\item{truth}{The column identifier for the true class results
(that is a \code{factor}). This should be an unquoted column name although
this argument is passed by expression and supports
\link[rlang:topic-inject]{quasiquotation} (you can unquote column
names). For \verb{_vec()} functions, a \code{factor} vector.}

\item{na_rm}{A \code{logical} value indicating whether \code{NA}
values should be stripped before the computation proceeds.}

\item{event_level}{A single string. Either \code{"first"} or \code{"second"} to specify
which level of \code{truth} to consider as the "event". This argument is only
applicable when \code{estimator = "binary"}. The default uses an internal helper
that defaults to \code{"first"}.}

\item{case_weights}{The optional column identifier for case weights.
This should be an unquoted column name that evaluates to a numeric column
in \code{data}. For \verb{_vec()} functions, a numeric vector,
\code{\link[hardhat:importance_weights]{hardhat::importance_weights()}}, or \code{\link[hardhat:frequency_weights]{hardhat::frequency_weights()}}.}
}
\value{
A tibble with class \code{gain_df} or \code{gain_grouped_df} having columns:
\itemize{
\item \code{.n} The index of the current sample.
\item \code{.n_events} The index of the current \emph{unique} sample. Values with repeated
\code{estimate} values are given identical indices in this column.
\item \code{.percent_tested} The cumulative percentage of values tested.
\item \code{.percent_found} The cumulative percentage of true results relative to the
total number of true results.
}

If using the \code{case_weights} argument, all of the above columns will be
weighted. This makes the most sense with frequency weights, which are integer
weights representing the number of times a particular observation should be
repeated.
}
\description{
\code{gain_curve()} constructs the full gain curve and returns a tibble. See
\code{\link[=gain_capture]{gain_capture()}} for the relevant area under the gain curve. Also see
\code{\link[=lift_curve]{lift_curve()}} for a closely related concept.
}
\details{
There is a \code{\link[ggplot2:autoplot]{ggplot2::autoplot()}} method for quickly visualizing the curve.
This works for binary and multiclass output, and also works with grouped data
(i.e. from resamples). See the examples.

The greater the area between the gain curve and the baseline, the better the
model.

Gain curves are identical to CAP curves (cumulative accuracy profile). See
the Engelmann reference for more information on CAP curves.
}
\section{Gain and Lift Curves}{


The motivation behind cumulative gain and lift charts is as a visual method to
determine the effectiveness of a model when compared to the results one
might expect without a model. As an example, without a model, if you were
to advertise to a random 10\% of your customer base, then you might expect
to capture 10\% of the of the total number of positive responses had you
advertised to your entire customer base. Given a model that predicts
which customers are more likely to respond, the hope is that you can more
accurately target 10\% of your customer base and capture
\code{>}10\% of the total number of positive responses.

The calculation to construct gain curves is as follows:
\enumerate{
\item \code{truth} and \code{estimate} are placed in descending order by the \code{estimate}
values (\code{estimate} here is a single column supplied in \code{...}).
\item The cumulative number of samples with true results relative to the
entire number of true results are found. This is the y-axis in a gain chart.
}
}

\section{Multiclass}{


If a multiclass \code{truth} column is provided, a one-vs-all
approach will be taken to calculate multiple curves, one per level.
In this case, there will be an additional column, \code{.level},
identifying the "one" column in the one-vs-all calculation.
}

\section{Relevant Level}{


There is no common convention on which factor level should
automatically be considered the "event" or "positive" result
when computing binary classification metrics. In \code{yardstick}, the default
is to use the \emph{first} level. To alter this, change the argument
\code{event_level} to \code{"second"} to consider the \emph{last} level of the factor the
level of interest. For multiclass extensions involving one-vs-all
comparisons (such as macro averaging), this option is ignored and
the "one" level is always the relevant result.
}

\examples{
# ---------------------------------------------------------------------------
# Two class example

# `truth` is a 2 level factor. The first level is `"Class1"`, which is the
# "event of interest" by default in yardstick. See the Relevant Level
# section above.
data(two_class_example)

# Binary metrics using class probabilities take a factor `truth` column,
# and a single class probability column containing the probabilities of
# the event of interest. Here, since `"Class1"` is the first level of
# `"truth"`, it is the event of interest and we pass in probabilities for it.
gain_curve(two_class_example, truth, Class1)

\dontshow{if (rlang::is_installed(c("ggplot2"))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# ---------------------------------------------------------------------------
# `autoplot()`

library(ggplot2)
library(dplyr)

# Use autoplot to visualize
# The top left hand corner of the grey triangle is a "perfect" gain curve
autoplot(gain_curve(two_class_example, truth, Class1))

# Multiclass one-vs-all approach
# One curve per level
hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  gain_curve(obs, VF:L) \%>\%
  autoplot()

# Same as above, but will all of the resamples
# The resample with the minimum (farthest to the left) "perfect" value is
# used to draw the shaded region
hpc_cv \%>\%
  group_by(Resample) \%>\%
  gain_curve(obs, VF:L) \%>\%
  autoplot()
\dontshow{\}) # examplesIf}
}
\references{
Engelmann, Bernd & Hayden, Evelyn & Tasche, Dirk (2003).
"Measuring the Discriminative Power of Rating Systems,"
Discussion Paper Series 2: Banking and Financial Studies 2003,01,
Deutsche Bundesbank.
}
\seealso{
Compute the relevant area under the gain curve with \code{\link[=gain_capture]{gain_capture()}}.

Other curve metrics: 
\code{\link{lift_curve}()},
\code{\link{pr_curve}()},
\code{\link{roc_curve}()}
}
\author{
Max Kuhn
}
\concept{curve metrics}
