






use crate::broadcast::BroadcastContext;
use crate::job::{ArcJob, HeapJob, JobFifo, JobRef};
use crate::latch::{CountLatch, Latch};
use crate::registry::{global_registry, in_worker, Registry, WorkerThread};
use crate::unwind;
use std::any::Any;
use std::fmt;
use std::marker::PhantomData;
use std::mem::ManuallyDrop;
use std::ptr;
use std::sync::atomic::{AtomicPtr, Ordering};
use std::sync::Arc;

#[cfg(test)]
mod test;





pub struct Scope<'scope> {
    base: ScopeBase<'scope>,
}






pub struct ScopeFifo<'scope> {
    base: ScopeBase<'scope>,
    fifos: Vec<JobFifo>,
}

struct ScopeBase<'scope> {
    
    
    registry: Arc<Registry>,

    
    
    panic: AtomicPtr<Box<dyn Any + Send + 'static>>,

    
    job_completed_latch: CountLatch,

    
    
    
    
    marker: PhantomData<Box<dyn FnOnce(&Scope<'scope>) + Send + Sync + 'scope>>,
}















/// # Example






/// # use rayon_core as rayon;
















/// # A note on threading






/// # Task execution
















/// # use rayon_core as rayon;




























































/// # Accessing stack data






/// # use rayon_core as rayon;























/// # use rayon_core as rayon;




















/// # use rayon_core as rayon;





















/// # use rayon_core as rayon;















/// # Panics









pub fn scope<'scope, OP, R>(op: OP) -> R
where
    OP: FnOnce(&Scope<'scope>) -> R + Send,
    R: Send,
{
    in_worker(|owner_thread, _| {
        let scope = Scope::<'scope>::new(Some(owner_thread), None);
        scope.base.complete(Some(owner_thread), || op(&scope))
    })
}








/// # Task execution







/// # use rayon_core as rayon;

















































/// For more details on this design, see Rayon [RFC #1].

/// [`breadth_first`]: struct.ThreadPoolBuilder.html#method.breadth_first
/// [RFC #1]: https://github.com/rayon-rs/rfcs/blob/master/accepted/rfc0001-scope-scheduling.md

/// # Panics









pub fn scope_fifo<'scope, OP, R>(op: OP) -> R
where
    OP: FnOnce(&ScopeFifo<'scope>) -> R + Send,
    R: Send,
{
    in_worker(|owner_thread, _| {
        let scope = ScopeFifo::<'scope>::new(Some(owner_thread), None);
        scope.base.complete(Some(owner_thread), || op(&scope))
    })
}












/// # Panics









pub fn in_place_scope<'scope, OP, R>(op: OP) -> R
where
    OP: FnOnce(&Scope<'scope>) -> R,
{
    do_in_place_scope(None, op)
}

pub(crate) fn do_in_place_scope<'scope, OP, R>(registry: Option<&Arc<Registry>>, op: OP) -> R
where
    OP: FnOnce(&Scope<'scope>) -> R,
{
    let thread = unsafe { WorkerThread::current().as_ref() };
    let scope = Scope::<'scope>::new(thread, registry);
    scope.base.complete(thread, || op(&scope))
}












/// # Panics









pub fn in_place_scope_fifo<'scope, OP, R>(op: OP) -> R
where
    OP: FnOnce(&ScopeFifo<'scope>) -> R,
{
    do_in_place_scope_fifo(None, op)
}

pub(crate) fn do_in_place_scope_fifo<'scope, OP, R>(registry: Option<&Arc<Registry>>, op: OP) -> R
where
    OP: FnOnce(&ScopeFifo<'scope>) -> R,
{
    let thread = unsafe { WorkerThread::current().as_ref() };
    let scope = ScopeFifo::<'scope>::new(thread, registry);
    scope.base.complete(thread, || op(&scope))
}

impl<'scope> Scope<'scope> {
    fn new(owner: Option<&WorkerThread>, registry: Option<&Arc<Registry>>) -> Self {
        let base = ScopeBase::new(owner, registry);
        Scope { base }
    }

    
    
    
    
    
    
    /// # Returns
    
    
    
    
    
    
    
    
    
    /// # Examples
    
    
    /// # use rayon_core as rayon;
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    /// # See also
    
    
    
    
    
    pub fn spawn<BODY>(&self, body: BODY)
    where
        BODY: FnOnce(&Scope<'scope>) + Send + 'scope,
    {
        let scope_ptr = ScopePtr(self);
        let job = HeapJob::new(move || unsafe {
            
            let scope = scope_ptr.as_ref();
            ScopeBase::execute_job(&scope.base, move || body(scope))
        });
        let job_ref = self.base.heap_job_ref(job);

        
        
        
        self.base.registry.inject_or_push(job_ref);
    }

    
    
    
    
    pub fn spawn_broadcast<BODY>(&self, body: BODY)
    where
        BODY: Fn(&Scope<'scope>, BroadcastContext<'_>) + Send + Sync + 'scope,
    {
        let scope_ptr = ScopePtr(self);
        let job = ArcJob::new(move || unsafe {
            
            let scope = scope_ptr.as_ref();
            let body = &body;
            let func = move || BroadcastContext::with(move |ctx| body(scope, ctx));
            ScopeBase::execute_job(&scope.base, func)
        });
        self.base.inject_broadcast(job)
    }
}

impl<'scope> ScopeFifo<'scope> {
    fn new(owner: Option<&WorkerThread>, registry: Option<&Arc<Registry>>) -> Self {
        let base = ScopeBase::new(owner, registry);
        let num_threads = base.registry.num_threads();
        let fifos = (0..num_threads).map(|_| JobFifo::new()).collect();
        ScopeFifo { base, fifos }
    }

    
    
    
    
    
    
    /// # See also
    
    
    
    
    
    /// [`Scope::spawn()`]: struct.Scope.html#method.spawn
    
    pub fn spawn_fifo<BODY>(&self, body: BODY)
    where
        BODY: FnOnce(&ScopeFifo<'scope>) + Send + 'scope,
    {
        let scope_ptr = ScopePtr(self);
        let job = HeapJob::new(move || unsafe {
            
            let scope = scope_ptr.as_ref();
            ScopeBase::execute_job(&scope.base, move || body(scope))
        });
        let job_ref = self.base.heap_job_ref(job);

        
        
        match self.base.registry.current_thread() {
            Some(worker) => {
                let fifo = &self.fifos[worker.index()];
                
                unsafe { worker.push(fifo.push(job_ref)) };
            }
            None => self.base.registry.inject(job_ref),
        }
    }

    
    
    
    
    pub fn spawn_broadcast<BODY>(&self, body: BODY)
    where
        BODY: Fn(&ScopeFifo<'scope>, BroadcastContext<'_>) + Send + Sync + 'scope,
    {
        let scope_ptr = ScopePtr(self);
        let job = ArcJob::new(move || unsafe {
            
            let scope = scope_ptr.as_ref();
            let body = &body;
            let func = move || BroadcastContext::with(move |ctx| body(scope, ctx));
            ScopeBase::execute_job(&scope.base, func)
        });
        self.base.inject_broadcast(job)
    }
}

impl<'scope> ScopeBase<'scope> {
    
    fn new(owner: Option<&WorkerThread>, registry: Option<&Arc<Registry>>) -> Self {
        let registry = registry.unwrap_or_else(|| match owner {
            Some(owner) => owner.registry(),
            None => global_registry(),
        });

        ScopeBase {
            registry: Arc::clone(registry),
            panic: AtomicPtr::new(ptr::null_mut()),
            job_completed_latch: CountLatch::new(owner),
            marker: PhantomData,
        }
    }

    fn heap_job_ref<FUNC>(&self, job: Box<HeapJob<FUNC>>) -> JobRef
    where
        FUNC: FnOnce() + Send + 'scope,
    {
        unsafe {
            self.job_completed_latch.increment();
            job.into_job_ref()
        }
    }

    fn inject_broadcast<FUNC>(&self, job: Arc<ArcJob<FUNC>>)
    where
        FUNC: Fn() + Send + Sync + 'scope,
    {
        let n_threads = self.registry.num_threads();
        let job_refs = (0..n_threads).map(|_| unsafe {
            self.job_completed_latch.increment();
            ArcJob::as_job_ref(&job)
        });

        self.registry.inject_broadcast(job_refs);
    }

    
    
    fn complete<FUNC, R>(&self, owner: Option<&WorkerThread>, func: FUNC) -> R
    where
        FUNC: FnOnce() -> R,
    {
        let result = unsafe { Self::execute_job_closure(self, func) };
        self.job_completed_latch.wait(owner);
        self.maybe_propagate_panic();
        result.unwrap() 
    }

    
    
    unsafe fn execute_job<FUNC>(this: *const Self, func: FUNC)
    where
        FUNC: FnOnce(),
    {
        let _: Option<()> = Self::execute_job_closure(this, func);
    }

    
    
    
    unsafe fn execute_job_closure<FUNC, R>(this: *const Self, func: FUNC) -> Option<R>
    where
        FUNC: FnOnce() -> R,
    {
        let result = match unwind::halt_unwinding(func) {
            Ok(r) => Some(r),
            Err(err) => {
                (*this).job_panicked(err);
                None
            }
        };
        Latch::set(&(*this).job_completed_latch);
        result
    }

    fn job_panicked(&self, err: Box<dyn Any + Send + 'static>) {
        
        if self.panic.load(Ordering::Relaxed).is_null() {
            let nil = ptr::null_mut();
            let mut err = ManuallyDrop::new(Box::new(err)); 
            let err_ptr: *mut Box<dyn Any + Send + 'static> = &mut **err;
            if self
                .panic
                .compare_exchange(nil, err_ptr, Ordering::Release, Ordering::Relaxed)
                .is_ok()
            {
                
            } else {
                
                let _: Box<Box<_>> = ManuallyDrop::into_inner(err);
            }
        }
    }

    fn maybe_propagate_panic(&self) {
        
        
        
        let panic = self.panic.swap(ptr::null_mut(), Ordering::Relaxed);
        if !panic.is_null() {
            let value = unsafe { Box::from_raw(panic) };
            unwind::resume_unwinding(*value);
        }
    }
}

impl<'scope> fmt::Debug for Scope<'scope> {
    fn fmt(&self, fmt: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt.debug_struct("Scope")
            .field("pool_id", &self.base.registry.id())
            .field("panic", &self.base.panic)
            .field("job_completed_latch", &self.base.job_completed_latch)
            .finish()
    }
}

impl<'scope> fmt::Debug for ScopeFifo<'scope> {
    fn fmt(&self, fmt: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt.debug_struct("ScopeFifo")
            .field("num_fifos", &self.fifos.len())
            .field("pool_id", &self.base.registry.id())
            .field("panic", &self.base.panic)
            .field("job_completed_latch", &self.base.job_completed_latch)
            .finish()
    }
}





struct ScopePtr<T>(*const T);


unsafe impl<T: Sync> Send for ScopePtr<T> {}


unsafe impl<T: Sync> Sync for ScopePtr<T> {}

impl<T> ScopePtr<T> {
    
    unsafe fn as_ref(&self) -> &T {
        &*self.0
    }
}
