co.intervals <- function (x, number = 6, overlap = 0.5)
{
    x <- sort(x[!is.na(x)])
    n <- length(x)
    ## "from the record"
    r <- n/(number * (1 - overlap) + overlap)
    ii <- 0:(number - 1) * (1 - overlap) * r
    cbind(x[round(1+ii)], x[round(r+ii)])
}

panel.smooth <- function(x, y, col = par("col"), pch = par("pch"),
			 col.smooth = "red", span = 2/3, iter = 3, ...)
{
    points(x, y, pch=pch, col=col)
    lines(lowess(x, y, f=span, iter=iter), col = col.smooth, ...)
}

coplot <- function (formula, data, given.values, panel=points, rows, columns,
		    show.given = TRUE, col = par("fg"), pch=par("pch"),
                    xlab = paste("Given :", a.name),
                    ylab = paste("Given :", b.name),
                    number = 6, overlap = 0.5, ...)
{
    deparen <- function(expr) {
	while (is.language(expr) && !is.name(expr) && deparse(expr[[1]])== "(")
	    expr <- expr[[2]]
	expr
    }
    bad.formula <- function() stop("invalid conditioning formula")
    bad.lengths <- function() stop("incompatible variable lengths")

    ## parse and check the formula

    formula <- deparen(formula)
    if (!inherits(formula, "formula"))
	bad.formula()
    y <- deparen(formula[[2]])
    rhs <- deparen(formula[[3]])
    if (deparse(rhs[[1]]) != "|")
	bad.formula()
    x <- deparen(rhs[[2]])
    rhs <- deparen(rhs[[3]])
    if (is.language(rhs) && !is.name(rhs)
	&& (deparse(rhs[[1]]) == "*" || deparse(rhs[[1]]) == "+")) {
	have.b <- TRUE
	a <- deparen(rhs[[2]])
	b <- deparen(rhs[[3]])
    } else {
	have.b <- FALSE
	a <- rhs
    }

    ## evaluate the formulae components to get the data values

    if (missing(data))
	data <- sys.frame(sys.parent())
    x.name <- deparse(x)
    x <- eval(x, data, sys.frame(sys.parent()))
    nobs <- length(x)
    y.name <- deparse(y)
    y <- eval(y, data, sys.frame(sys.parent()))
    if(length(y) != nobs) bad.lengths()
    a.name <- deparse(a)
    a <- eval(a, data, sys.frame(sys.parent()))
    if(length(a) != nobs) bad.lengths()
    if(is.character(a)) a <- as.factor(a)
    if (have.b) {
	b.name <- deparse(b)
	b <- eval(b, data, sys.frame(sys.parent()))
	if(length(b) != nobs) bad.lengths()
        if(is.character(b)) b <- as.factor(b)
        missingrows <- which(is.na(x) | is.na(y) | is.na(a) | is.na(b))
    }
    else {
        missingrows <- which(is.na(x) | is.na(y) | is.na(a))
        b <- NULL
    }

    ## generate the given value intervals

    number <- as.integer(number)
    if(length(number)==0 || any(number < 1)) stop("number must be integer >= 1")
    if(any(overlap >= 1)) stop("overlap must be < 1 (and typically >= 0).")

    bad.givens <- function() stop("invalid given.values")
    if(missing(given.values)) {
	a.intervals <-
            if(is.factor(a)) {
                i <- 1:nlevels(a)
                a <- as.numeric(a)
                cbind(i,i)
            } else co.intervals(a,number=number[1],overlap=overlap[1])
	b.intervals <-
            if (have.b) {
                if(is.factor(b)) {
                    i <- 1:nlevels(b)
                    b <- as.numeric(b)
                    cbind(i,i)
                }
                else {
                    if(length(number)==1) number  <- rep(number,2)
                    if(length(overlap)==1)overlap <- rep(overlap,2)
                    co.intervals(b,number=number[2],overlap=overlap[2])
                }
            }
    } else {
	if(!is.list(given.values))
	    given.values <- list(given.values)
	if(length(given.values) != (if(have.b) 2 else 1))
	    bad.givens()
	a.intervals <- given.values[[1]]
	if(is.factor(a)) {
	    a.intervals <-
                if(is.character(a.intervals))
                    match(a.intervals, levels(a))
                else cbind(a.intervals, a.intervals)
	    a <- as.numeric(a)
	}
        else if(is.numeric(a)) {
	    if(!is.numeric(a.intervals)) bad.givens()
	    if(!is.matrix(a.intervals) || ncol(a.intervals) != 2)
		a.intervals <- cbind(a.intervals, a.intervals)
	}
	if(have.b) {
	    b.intervals <- given.values[[2]]
	    if(is.factor(b)) {
		b.intervals <-
                    if(is.character(b.intervals))
                        match(b.intervals, levels(b))
                    else cbind(b.intervals, b.intervals)
		b <- as.numeric(b)
	    }
            else if(is.numeric(b)) {
		if(!is.numeric(b.intervals)) bad.givens()
		if(!is.matrix(b.intervals) || ncol(b.intervals) != 2)
		    b.intervals <- cbind(b.intervals, b.intervals)
	    }
	}
    }
    if(any(is.na(a.intervals)) || (have.b && any(is.na(b.intervals))))
        bad.givens()

    ## compute the page layout

    if (have.b) {
	rows    <- nrow(b.intervals)
	columns <- nrow(a.intervals)
	nplots <- rows * columns
        if(length(show.given) < 2) show.given <- rep(show.given, 2)
    } else {
	nplots <- nrow(a.intervals)
	if (missing(rows)) {
	    if (missing(columns)) {
		rows <- ceiling(round(sqrt(nplots)))
		columns <- ceiling(nplots/rows)
	    }
	    else rows <- ceiling(nplots/columns)
	}
	else if (missing(columns))
	    columns <- ceiling(nplots/rows)
	if (rows * columns < nplots)
	    stop("rows * columns too small")
    }
    total.columns <- columns
    total.rows <- rows
    f.col <- f.row <- 1
    if(show.given[1]) {
        total.rows <- rows + 1
        f.row <- rows/total.rows
    }
    if(have.b && show.given[2]) {
        total.columns <- columns + 1
        f.col <- columns/total.columns
    }

    ## Start Plotting only now

    opar <- par(mfrow = c(total.rows, total.columns),
		oma = if(have.b) rep(5, 4) else c(5, 6, 5, 4),
		mar = if(have.b) rep(0, 4) else c(0.5, 0, 0.5, 0),
		new = FALSE)
    on.exit(par(opar))
    plot.new()
    xlim <- range(x[is.finite(x)])
    ylim <- range(y[is.finite(y)])
    pch <- rep(pch, length=nobs)
    col <- rep(col, length=nobs)
    do.panel <- function(index) {
        ## Use `global' variables
        ##	id;     rows, columns,  total.rows, total.columns, nplots
        ##		xlim, ylim
	istart <- (total.rows - rows) + 1
	i <- total.rows - ((index - 1)%/%columns)
	j <- (index - 1)%%columns + 1
	par(mfg = c(i, j, total.rows, total.columns))
	plot.new()
	plot.window(xlim, ylim, log = "")
        if(any(is.na(id))) id[is.na(id)] <- FALSE
	if(any(id)) {
	    grid(lty="solid")
	    panel(x[id], y[id], col = col[id], pch=pch[id], ...)
	}
	if((i == total.rows) && (j%%2 == 0))
	    axis(1)
	else if((i == istart || index + columns > nplots) && (j%%2 == 1))
	    axis(3)
	if((j == 1) && ((total.rows - i)%%2 == 0))
	    axis(2)
	else if((j == columns || index == nplots) && ((total.rows - i)%%2 == 1))
	    axis(4)
	## if(i == total.rows)
	##	axis(1, labels = (j%%2 == 0))
	## if(i == istart || index + columns > nplots)
	##	axis(3, labels = (j%%2 == 1))
	## if(j == 1)
	##	axis(2, labels = ((total.rows - i)%%2 == 0))
	## if(j == columns || index == nplots)
	##	axis(4, labels = ((total.rows - i)%%2 == 1))
	box()
    }## END function do.panel()

    if(have.b) {
	count <- 1
	for(i in 1:rows) {
	    for(j in 1:columns) {
		id <- ((a.intervals[j,1] <= a) & (a <= a.intervals[j,2]) &
		       (b.intervals[i,1] <= b) & (b <= b.intervals[i,2]))
		do.panel(count)
		count <- count + 1
	    }
	}
    } else {
	for (i in 1:nplots) {
	    id <- ((a.intervals[i,1] <= a) & (a <= a.intervals[i,2]))
	    do.panel(i)
	}
    }
    mtext(x.name, side=1, at=0.5*f.col, outer=TRUE, line=3.5, xpd=TRUE)
    mtext(y.name, side=2, at=0.5*f.row, outer=TRUE, line=3.5, xpd=TRUE)

    if(show.given[1]) {
	mar <- par("mar")
	nmar <- mar + c(4,0,0,0)
	par(fig = c(0, f.col, f.row, 1), mar = nmar, new=TRUE)
	plot.new()
	nint <- nrow(a.intervals)
	plot.window(range(a.intervals[is.finite(a.intervals)]),
                    .5+c(0, nint), log="")
	rect(a.intervals[,1], 1:nint-0.3,
	     a.intervals[,2], 1:nint+0.3, col=gray(0.9))
	axis(3)
	axis(1, labels=FALSE)
	box()
	mtext(xlab, side=3, at=mean(par("usr")[1:2]), line=3, xpd=TRUE)
    }
    else { ## i. e. !show.given
        mtext(xlab, side=3, at= 0.5*f.col, line= 3.25, outer= TRUE, xpd=TRUE)
    }
    if(have.b) {
        if(show.given[2]) {
            nmar <- mar + c(0, 4, 0, 0)
	    par(fig = c(f.col, 1, 0, f.row), mar = nmar, new=TRUE)
	    plot.new()
	    nint <- nrow(b.intervals)
	    plot.window(.5+c(0, nint),
			range(b.intervals, finite=TRUE), log="")
	    rect(1:nint-0.3, b.intervals[,1],
		 1:nint+0.3, b.intervals[,2], col=gray(0.9))
	    axis(4)
	    axis(2, labels=FALSE)
	    box()
	    mtext(ylab, side=4, at=mean(par("usr")[3:4]), line=3, xpd=TRUE)
	}
        else {
            mtext(ylab, side=4, at=0.5*f.row, line= 3.25, outer=TRUE, xpd=TRUE)
        }
    }
    if (length(missingrows) > 0) {
        cat("\nMissing rows:",missingrows,"\n")
        invisible(missingrows)
    }
}
