#' Parallel Vorob'ev criterion
#'
#' Evaluation of the Vorob'ev criterion for candidate points \code{x}, assuming that some other points are also going to be evaluated. To be used in optimization routines, like in \code{\link{max_vorob_parallel_gpc}}. To avoid numerical instabilities, the new points are evaluated only if they are not too close to an existing observation, or if there is some observation noise. The criterion is the integral of the posterior Vorob'ev uncertainty.
#'
#' @param x input vector of size d at which one wants to evaluate the criterion.
#' @param other.points vector giving the other \code{batchsize-1} points at which one wants to evaluate the criterion.
#' @param integration.points p*d matrix of points for numerical integration in the design space.
#' @param integration.weights vector of size p corresponding to the weights of these integration points.
#' @param intpoints.oldmean vector of size p corresponding to the latent GP mean at the integration points before adding \code{x} to the design of experiments.
#' @param intpoints.oldsd vector of size p corresponding to the latent GP standard deviation at the integration points before adding \code{x} to the design of experiments.
#' @param precalc.data list containing precalculated data. This list can be generated using the \code{precomputeUpdateData} function. 
#' @param object object of class \code{\link[GPCsign]{gpcm}}.
#' @param new.noise.var optional scalar value of the noise variance of the new observations.
#' @param batchsize number of points to sample simultaneously. The sampling criterion will return batchsize points at a time for sampling.
#' @param alpha a scalar representing the Vorob'ev threshold.
#' @param current.vorob current value of the vorob criterion (before adding new observations).
#' @param seed to fix the seed.
#'
#' @return Parallel Vorob'ev value
#' @seealso [max_vorob_parallel_gpc()]
#' @export
#'
#' @references Menz, M., Munoz-Zuniga, M., Sinoquet, D. Estimation of simulation failure set with active learning based on Gaussian Process classifiers and random set theory (2023). \url{https://hal.science/hal-03848238}.
#' @references Chevalier, C. Fast uncertainty reduction strategies relying on Gaussian process models PhD Thesis. University of Bern (2013).
#' @references Bachoc, F., Helbert, C. & Picheny, V. Gaussian process optimization with failures: classification and convergence proof. \emph{J Glob Optim} \bold{78}, 483–506 (2020). \doi{10.1007/s10898-020-00920-0}.
#'
#' @author Morgane MENZ, Delphine SINOQUET, Miguel MUNOZ-ZUNIGA. Contributors: Naoual SERRAJI.
#'
vorob_optim_parallel2_gpc <- function(x,other.points, integration.points,integration.weights=NULL,
		intpoints.oldmean,intpoints.oldsd,precalc.data,
		object, new.noise.var=NULL,batchsize,alpha,current.vorob,
		 seed=NULL){

	x.complete <- c(x,other.points)
	return(vorob_optim_parallel_gpc(
		x = x.complete, integration.points = integration.points, integration.weights = integration.weights,
		intpoints.oldmean = intpoints.oldmean,intpoints.oldsd = intpoints.oldsd,precalc.data = precalc.data,
		object = object,new.noise.var = new.noise.var,batchsize=batchsize,alpha=alpha,
		current.vorob=current.vorob, seed=seed
		)
	)

}
