\name{AdMitMH}
\alias{AdMitMH}
\concept{Mixture of Student-t Distributions}
\concept{Adaptive mixture}
\concept{Metropolis-Hastings}
\title{Independence Chain Metropolis-Hastings Algorithm using an Adaptive
  Mixture of Student-t Distributions as the Candidate Density
}
\description{Performs independence chain Metropolis-Hastings (M-H) sampling using
  an adaptive mixture of Student-t distributions as the candidate density
}
\usage{%
AdMitMH(N = 1e5, KERNEL, mit = list(), ...)
}
\arguments{%
  \item{N}{number of draws generated by the independence chain M-H algorithm (positive
    integer number). Default: \code{N = 1e5}.}
  \item{KERNEL}{kernel function of the target density on which the adaptive mixture is fitted. This
    function should be vectorized for speed purposes (i.e., its first
    argument should be a matrix and its output a vector). Moreover, the function must contain
    the logical argument \code{log}. If \code{log = TRUE}, the function
    returns (natural) logarithm values of kernel function. \code{NA}
    and \code{NaN} values are not allowed. (See the function
    \code{\link{AdMit}} for examples of \code{KERNEL} implementation.)}
  \item{mit}{list containing information on the mixture approximation (see *Details*).}
  \item{\ldots}{further arguments to be passed to \code{KERNEL}.}
}
\details{%
  The argument \code{mit} is a list containing information on the
  adaptive mixture of Student-t distributions. The following components must
  be provided:
  
  \describe{%
    \item{\code{p}}{vector (of length \eqn{H}) of mixing probabilities.}
    \item{\code{mu}}{matrix (of size \eqn{H \times d}{Hxd}) containing
      the vectors of modes (in row) of the mixture components.}
    \item{\code{Sigma}}{matrix (of size \eqn{H \times d^2}{Hxd*d})
      containing the scale matrices (in row) of the mixture components.}
    \item{\code{df}}{degrees of freedom parameter of the Student-t
      components (real number not smaller than one).}
  }
  
  where \eqn{H (\geq 1)}{H (>=1)} is the number of components and
  \eqn{d (\geq 1)}{d (>=1)} is the dimension of the first argument in \code{KERNEL}. Typically,
  \code{mit} is estimated by the function \code{\link{AdMit}}.
}
\value{%
  A list with the following components:\cr
  
  \code{draws}: matrix (of size \code{N}\eqn{\times d}{xd}) of draws
  generated by the independence chain M-H algorithm,
  where \code{N} \eqn{(\geq 1)}{(>=1)} is the number of draws
  and \eqn{d (\geq 1)}{d (>=1)} is the
  dimension of the first argument in \code{KERNEL}.\cr

  \code{accept}: acceptance rate of the independence chain M-H algorithm.
}
\note{%
  Further details and examples of the \R package \code{AdMit}
  can be found in Ardia, Hoogerheide and van Dijk (2009a,b). See also
  the package vignette by typing \code{vignette("AdMit")} and the
  files \file{AdMitJSS.txt} and \file{AdMitRnews.txt} in the \file{/doc} package's folder.

  Further information on the Metropolis-Hastings algorithm can be found
  in Chib and Greenberg (1995) and Koop (2003).
  
  Please cite the package in publications. Use \code{citation("AdMit")}.
}
\author{David Ardia}
\references{%
  Ardia, D., Hoogerheide, L.F., van Dijk, H.K. (2009a).
  AdMit: Adaptive Mixture of Student-t Distributions.
  \emph{The R Journal} \bold{1}(1), pp.25--30.
  \url{http://journal.r-project.org/2009-1/}

  Ardia, D., Hoogerheide, L.F., van Dijk, H.K. (2009b).
  Adaptive Mixture of Student-t Distributions as a Flexible Candidate
  Distribution for Efficient Simulation: The R Package AdMit.
  \emph{Journal of Statistical Software} \bold{29}(3), pp.1--32.
  \url{http://www.jstatsoft.org/v29/i03/}

  Chib, S., Greenberg, E. (1995). 
  Understanding the Metropolis-Hasting Algorithm. 
  \emph{The American Statistician} \bold{49}(4), pp.327--335.

  Koop, G. (2003). 
  \emph{Bayesian Econometrics}. 
  Wiley-Interscience (London, UK). 
  ISBN: 0470845678.
}
\seealso{%
  \code{\link{AdMitIS}} for importance sampling using an adaptive
  mixture of Student-t distributions as the importance density,
  \code{\link{AdMit}} for fitting
  an adaptive mixture of Student-t distributions to a target density
  through its \code{KERNEL} function; the package coda for MCMC output
  analysis.
}
\examples{%
  ## NB : Low number of draws for speedup. Consider using more draws!
  ## Gelman and Meng (1991) kernel function
  GelmanMeng <- function(x, A = 1, B = 0, C1 = 3, C2 = 3, log = TRUE)
  {
    if (is.vector(x))
      x <- matrix(x, nrow = 1)
      r <- -.5 * (A * x[,1]^2 * x[,2]^2 + x[,1]^2 + x[,2]^2
                - 2 * B * x[,1] * x[,2] - 2 * C1 * x[,1] - 2 * C2 * x[,2])
      if (!log)
        r <- exp(r)
      as.vector(r)
    }

  ## Run the AdMit function to fit the mixture approximation
  set.seed(1234)
  outAdMit <- AdMit(KERNEL = GelmanMeng, 
                    mu0 = c(0.0, 0.1), control = list(Ns = 1e4))

  ## Run M-H using the mixture approximation as the candidate density
  outAdMitMH <- AdMitMH(N = 1e4, KERNEL = GelmanMeng, mit = outAdMit$mit)
  options(digits = 4, max.print = 40)
  print(outAdMitMH)

  ## Use functions provided by the package coda to obtain
  ## quantities of interest for the density whose kernel is 'GelmanMeng'
  library("coda")
  draws <- as.mcmc(outAdMitMH$draws)
  draws <- window(draws, start = 1001)
  colnames(draws) <- c("X1", "X2")
  summary(draws)
  summary(draws)$stat[,3]^2 / summary(draws)$stat[,4]^2 ## RNE
  plot(draws)
}
\keyword{htest}
