% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/depth.R
\name{depth}
\alias{depth}
\title{Depth calculation}
\usage{
depth(u, X, method = "Projection", threads = -1, ...)
}
\arguments{
\item{u}{Numerical vector or matrix whose depth is to be calculated. Dimension has to be the same as that of the observations.}

\item{X}{The data as a matrix, data frame or list. If it is a matrix or data frame, then each row is viewed as one multivariate observation. If it is a list, all components must be numerical vectors of equal length (coordinates of observations).}

\item{method}{Character string which determines the depth function. \code{method} can be "Projection" (the default), "Mahalanobis", "Euclidean" or "Tukey". For details see \code{\link{depth}}.}

\item{threads}{number of threads used in parallel computations. Default value -1 means that all possible cores will be used.}

\item{...}{parameters specific to method --- see \code{\link{depthEuclid}}}
}
\description{
Calculate depth functions.
}
\details{
{The Mahalanobis depth} \deqn{ {D}_{MAH}(y, {X} ^ {n}) = \frac{ 1 }{ 1 + {{(y - \bar{x})} ^ {T}}{{S} ^ {-1}}(y - \bar{x}) }, } where \eqn{ S } denotes the sample covariance matrix \eqn{ {X} ^ {n} }.

A symmetric projection depth \eqn{ D\left( x, X\right) } of a point \eqn{ x \in {{{R}} ^ {d}} }, \eqn{ d \ge 1 } is defined as \eqn{ D\left( x, X\right)_{PRO} = {{\left[ 1 + su{{p}_{\left\| u \right\| = 1}}\frac{ \left| {{u} ^ {T}}x - Med\left( {{u} ^ {T}}X\right)\right| }{ MAD\left( {{u} ^ {T}}X\right) }\right]} ^ {-1}}, } where Med denotes the univariate median, \eqn{ MAD\left( Z \right) } = \eqn{ Med\left(\left| Z - Med\left( Z \right)\right|\right) }. Its sample version denoted by \eqn{ D\left( x, {X} ^ {n} \right) } or \eqn{ D\left( x, {X} ^ {n} \right) } is obtained by replacing \eqn{ F } by its empirical counterpart \eqn{ {{F}_{n}} } calculated from the sample \eqn{ {X} ^ {n} } .

Next interesting depth is the weighted \eqn{ {L} ^ {p} } depth. The weighted \eqn{ {L} ^ {p} } depth \eqn{ D({x}, F) } of a point \eqn{ {x} \in {R} ^ {d} }, \eqn{ d \ge 1 } generated by \eqn{ d } dimensional random vector \eqn{ {X} } with distribution \eqn{ F }, is defined as \eqn{ D({x}, F) = \frac{1 }{ 1 + Ew({{\left\| x - X \right\| }_{p}}) }, } where \eqn{ w } is a suitable weight function on \eqn{ [0, \infty) }, and \eqn{ {{\left\| \cdot \right\| }_{p}} } stands for the \eqn{ {L} ^ {p} } norm (when p = 2 we have usual Euclidean norm). We assume that \eqn{ w } is non-decreasing and continuous on \eqn{ [0, \infty) } with \eqn{ w(\infty-) = \infty }, and for \eqn{ a, b \in {{{R}} ^ {d}} } satisfying \eqn{ w(\left\| a + b \right\|) \le w(\left\| a \right\|) + w(\left\| b \right\|) }. Examples of the weight functions are: \eqn{ w(x) = a + bx }, \eqn{ a, b > 0 } or \eqn{ w(x) = {x} ^ {\alpha} }. The empirical version of the weighted \eqn{ {L} ^ {p} } depth is obtained by replacing distribution \eqn{ F } of \eqn{ {X} } in \eqn{ Ew({{\left\| {x} - {X} \right\| }_{p}}) = \int {w({{\left\| x - t \right\| }_{p}})}dF(t) } by its empirical counterpart calculated from the sample \eqn{ {{{X}} ^ {n}} }...

The Projection and Tukey's depths are calculated using an approximate algorithm. Calculations of Mahalanobis, Euclidean and \eqn{ L ^ p } depths are exact. Returns the depth of multivariate point u with respect to data set X.
}
\examples{
library(robustbase)

# Calculation of Projection depth
data(starsCYG, package = "robustbase")
depth(t(colMeans(starsCYG)), starsCYG)

# Also for matrices
depth(starsCYG, starsCYG)

# Projection depth applied to a large bivariate data set
x <- matrix(rnorm(9999), nc = 3)
depth(x, x)

}
\references{
Liu, R.Y., Parelius, J.M. and Singh, K. (1999), Multivariate analysis by data depth: Descriptive statistics, graphics and inference (with discussion), Ann. Statist., 27, 783--858.

Mosler K (2013). Depth statistics. In C Becker, R Fried, K S (eds.), Robustness and Complex Data Structures, Festschrift in Honour of Ursula Gather, pp. 17--34. Springer.

Rousseeuw, P.J. and Struyf, A. (1998), Computing location depth and regression depth in higher dimensions, Stat. Comput., 8, 193--203.

Zuo, Y. and Serfling, R. (2000), General Notions of Statistical Depth Functions, Ann. Statist., 28, no. 2, 461--482.
}
\seealso{
\code{\link{depthContour}} and \code{\link{depthPersp}} for depth graphics.
}
\author{
Daniel Kosiorowski, Mateusz Bocian, Anna Wegrzynkiewicz and Zygmunt Zawadzki from Cracow University of Economics.
}
\keyword{depth}
\keyword{function}
\keyword{multivariate}
\keyword{nonparametric}
\keyword{robust}
