% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/QAIC.R
\name{QAIC}
\alias{QAIC}
\alias{chi2_select}
\title{Tools for model selection when distance sampling data are overdispersed}
\usage{
QAIC(object, ..., chat = NULL, k = 2)

chi2_select(object, ...)
}
\arguments{
\item{object}{a fitted detection function object}

\item{\dots}{additional fitted model objects.}

\item{chat}{a value of \eqn{\hat{c}}{chat} to be used in QAIC calculation}

\item{k}{penalty per parameter to be used; default 2}
}
\value{
a \code{data.frame} with one row per model supplied, in the same order as
given
}
\description{
Overdispersion causes AIC to select overly-complex models, so analysts
should specify the number/order of adjustment terms manually when fitting
distance sampling models to data from camera traps, rather than allowing
automated selection using AIC. Howe et al (2019) described a two-step method
for selecting among models of the detection function in the face of
overdispersion.
}
\details{
In step 1, and overdispersion factor (\eqn{\hat{c}}{chat}) is computed
either (1) for each key function family, from the most complex model in each
family, as the chi-square goodness of fit test statistic divided by its
degrees of freedom (\eqn{\hat{c}_1}{chat1}), or (2) for all models in the
candidate set, from the raw data (\eqn{\hat{c}_1}{chat2}). In camera trap
surveys of solitary animals, \eqn{\hat{c}_1}{chat2} would be the mean number
of distance observations recorded during a single pass by an animal in front
of a trap. In surveys of social animals employing human observers,
\eqn{\hat{c}_1}{chat2} would be the mean number of detected animals per
detected group, and in camera trap surveys of social animals
\eqn{\hat{c}_1}{chat2} the mean number of distance observations recorded
during an encounter between a group of animals and a CT.  In step two, the
chi-square goodness of fit statistic divided by its degrees of freedom is
calculated for the QAIC-minimizing model within each key function, and the
model with the lowest value is selected for estimation.

The \code{QAIC()} function should only be used select among models with the same
key function (step 1). \code{QAIC()} uses \eqn{\hat{c}_1}{chat1} by default,
computing it from the model with the most parameters. Alternatively,
\eqn{\hat{c}_1}{chat2} can be calculated from the raw data and included in
the call to \code{QAIC()}. Users must identify the QAIC-minimizing model within
key functions from the resulting \code{data.frame}, and provide these models as
arguments in \code{ch2_select()}. \code{chi2_select()} then computes and reports the
chi-square goodness of fit statistic divided by its degrees of freedom for
each of those models. The model with the lowest value is recommended for
estimation.
}
\examples{
\dontrun{
library(Distance)
data("wren_cuecount")

# fit hazard-rate key models
w3.hr0 <- ds(wren_cuecount, transect="point", key="hr", adjustment=NULL,
             truncation=92.5)
w3.hr1 <- ds(wren_cuecount, transect="point", key="hr", adjustment="cos",
             order=2, truncation=92.5)
w3.hr2 <- ds(wren_cuecount, transect="point", key="hr", adjustment="cos",
             order=c(2, 4), truncation=92.5)

# fit unform key models
w3.u1 <- ds(wren_cuecount, transect="point", key="unif", adjustment="cos",
            order=1, truncation=92.5)
w3.u2 <- ds(wren_cuecount, transect="point", key="unif", adjustment="cos",
            order=c(1,2), monotonicity="none",  truncation=92.5)
w3.u3 <- ds(wren_cuecount, transect="point", key="unif", adjustment="cos",
            order=c(1,2,3), monotonicity="none", truncation=92.5)

# fit half-normal key functions
w3.hn0 <- ds(wren_cuecount, transect="point", key="hn", adjustment=NULL,
             truncation=92.5)
w3.hn1 <- ds(wren_cuecount, transect="point", key="hn", adjustment="herm",
             order=2, truncation=92.5)

# stage 1: calculate QAIC per model set
QAIC(w3.hr0, w3.hr1, w3.hr2)  # no adjustments smallest
QAIC(w3.u1, w3.u2, w3.u3)     # 2 adjustment terms (by 0.07)
QAIC(w3.hn0, w3.hn1)  # no adjustments smallest

# stage 2: select using chi^2/degrees of freedom between sets
chi2_select(w3.hr0, w3.u2, w3.hn0)

# example using a pre-calculated chat
chat <- attr(QAIC(w3.hr0, w3.hr1, w3.hr2), "chat")
QAIC(w3.hr0, chat=chat)
}
}
\references{
Howe, E. J., Buckland, S. T., Després-Einspenner, M.-L., & Kühl, H. S. (2019). Model selection with overdispersed distance sampling data. Methods in Ecology and Evolution, 10(1), 38–47. \doi{10.1111/2041-210X.13082}
}
\author{
David L Miller, based on code from Eric Rexstad and explanation from
Eric Howe.
}
