% Generated by roxygen2 (4.0.2): do not edit by hand
\name{theta-utils}
\alias{check_theta}
\alias{complete_theta}
\alias{flatten_theta}
\alias{get_initial_theta}
\alias{get_theta_bounds}
\alias{theta-utils}
\alias{theta2tau}
\alias{theta2unbounded}
\alias{unflatten_theta}
\title{Utilities for the parameter vector of Lambert W\eqn{\times} F distributions}
\usage{
check_theta(theta, distname)

complete_theta(theta = list(), LambertW.input = NULL)

flatten_theta(theta)

get_initial_theta(y, distname, type = c("h", "hh", "s"),
  theta.fixed = list(alpha = 1), method = c("Taylor", "IGMM"))

get_theta_bounds(distname, beta, type = c("s", "h", "hh"),
  not.negative = FALSE)

theta2tau(theta = list(beta = c(0, 1)), distname)

theta2unbounded(theta, distname, type = c("h", "hh", "s"), inverse = FALSE)

unflatten_theta(theta.flattened, distname, type)
}
\arguments{
\item{theta}{list; a (possibly incomplete) list of parameters \code{alpha}, \code{beta}, \code{gamma},
\code{delta}. \code{\link{complete_theta}} fills in default values for missing entries.}

\item{distname}{string; name of input distribution; see \code{\link{get_distnames}}.}

\item{LambertW.input}{optional; if \code{beta} is missing in \code{theta},
\code{LambertW.input} (which has a \code{beta} element) must be specified.}

\item{y}{a numeric vector of real values (the observed data).}

\item{type}{type of Lambert W \eqn{\times} F distribution: skewed \code{"s"};
heavy-tail \code{"h"}; or skewed heavy-tail \code{"hh"}.}

\item{theta.fixed}{list; fixed parameters for the optimization; default
only \code{alpha = 1}.}

\item{method}{string; should a fast \code{"Taylor"} (default) approximation be used
(\code{\link{delta_Taylor}} or \code{\link{gamma_Taylor}}) to estimate \eqn{\delta} or
\eqn{\gamma}, or should \code{"IGMM"} (\code{\link{IGMM}}) estimates be used.
Use \code{"Taylor"} as initial values for \code{\link{IGMM}};
\code{\link{IGMM}} improves upon it and should be used for \code{\link{MLE_LambertW}}.
Do \strong{not} use \code{"IGMM"} as initial values for \code{\link{IGMM}} --
this will run \code{\link{IGMM}} twice.}

\item{beta}{vector; parameter \eqn{\boldsymbol \beta} of the input distribution.
See \code{\link{check_beta}} on how to specify \code{beta} for each distribution.}

\item{not.negative}{logical; if \code{TRUE} it sets the lower bounds for \code{alpha}
and \code{delta} to \code{0}.  Default: \code{FALSE}.}

\item{inverse}{logical; if \code{TRUE}, it transforms the unbounded \code{theta} back to
the original, bounded space. Default: \code{FALSE}.}

\item{theta.flattened}{named vector; flattened version of list \code{theta}.}
}
\value{
\code{check_theta} throws an error if list \code{theta} does not
define a proper Lambert W \eqn{\times} F distribution;
does nothing otherwise.

\code{complete_theta} returns a list containing:
\item{alpha}{ heavy tail exponent(s),}
\item{beta}{ named vector \eqn{\boldsymbol \beta} of the input distribution, }
\item{gamma}{ skewness parameter,}
\item{delta}{ heavy-tail parameter(s).}

\code{get_initial_theta} returns a list containing:
\item{alpha}{ heavy tail exponent; default: \code{1}, }
\item{beta}{ named vector \eqn{\boldsymbol \beta} of the input distribution;
estimated from the recovered input data \eqn{\widehat{\mathbf{x}}_{\widehat{\tau}}}, }
\item{gamma}{ skewness parameter; if \code{type} is \code{"h"} or \code{"hh"} \code{gamma = 0};
estimated from \code{\link{IGMM}}, }
\item{delta}{ heavy-tail parameter;
estimated from \code{\link{IGMM}}. If \code{type = "s"}, then \code{delta = 0}. }

\code{get_theta_bounds} returns a list containing two vectors:
\item{lower}{ flattened vector of lower bounds for valid \eqn{\theta}, }
\item{upper}{ flattened vector of upper bounds for valid \eqn{\theta}. }
}
\description{
These functions work with \eqn{\boldsymbol \theta = (\boldsymbol \beta, \gamma, \delta, \alpha)},
which fully parametrizes Lambert W\eqn{\times} F distributions.

See Details for more background information on some functions.

\code{check_theta} checks if \eqn{\theta = (\alpha, \boldsymbol \beta, \gamma, \delta)}
describes a well-defined Lambert W distribution.

\code{complete_theta} completes missing values in a parameters list so users
don't have to specify everything in detail. If not supplied, then the
following will be set: \code{alpha = 1}, \code{gamma = 0}, and \code{delta =
0}.

\code{flatten_theta} and \code{unflatten_theta} convert between the list
\code{theta} and its vector-style flattened type.  The flattened version is required
for several optimization routines, since they optimize over multivariate vectors -- not lists.

\code{get_initial_theta} provides initial estimates for \eqn{\alpha},
\eqn{\boldsymbol \beta}, \eqn{\gamma}, and \eqn{\delta}, which are then used in
maximum likelihood (ML) estimation (\code{\link{MLE_LambertW}}).

\code{get_theta_bounds} returns lower and upper bounds for \eqn{\theta}
(necessary for optimization such as \code{\link{MLE_LambertW}}).

\code{theta2tau} converts \eqn{\theta} to the transformation vector
\eqn{\tau = (\mu_x, \sigma_x, \gamma, \delta, \alpha)}.

\code{theta2unbounded} transforms \eqn{\theta} from the bounded space to an
unrestricted space (by \eqn{\log}-transformation on
\eqn{\sigma_x}, \eqn{\delta}, and \eqn{\alpha}; note that this restricts
\eqn{\gamma \geq 0}, \eqn{\delta \geq 0}, and \eqn{\alpha \geq 0}.).
}
\details{
\code{get_initial_theta} obtains a quick initial estimate of \eqn{\theta}
 by first finding the (approximate) input \eqn{\widehat{\boldsymbol x}_{\widehat{\theta}}}
by \code{\link{IGMM}}, and then estimating \eqn{\boldsymbol \beta} for this input data
\eqn{\widehat{\boldsymbol x}_{\widehat{\theta}} \sim F_X(x \mid \boldsymbol
\beta)} (see \code{\link{estimate_beta}}).

Converting \code{theta} to an unbouded space is especially useful for
optimization routines (like \code{\link[stats]{nlm}}),
which can be performed over an unconstrained space. The obtained optimum
can be converted back to the original space using the inverse transformation (set
\code{inverse = TRUE} transforms it via \eqn{\exp}) -- this guarantees that
the estimate satisfies non-negativity constraints
(if required). The main advantage is that this avoids using optimization routines with
boundary constraints -- since they are very slow compared to uncostrained optimization.
}
\examples{
\dontrun{
check_theta(theta = list(beta= c(1, 1, -1)), distname = "t")
}

check_theta(theta = list(beta= c(1, 1)), distname = "normal") # ok

params <- list(beta = c(2, 1), delta = 0.3) # alpha and gamma are missing
complete_theta(params) # added default values

params <- list(beta = c(2, 1), delta = 0.3, alpha = c(1, 2))
params <- complete_theta(params)
check_theta(params, distname = 'normal')

###
x <- rnorm(1000)
get_initial_theta(x, distname="normal", type="h")
get_initial_theta(x, distname="normal", type="s")

# starting values for the skewed version of a Normal
y <- rLambertW(n=1000, distname="exp", beta = 2, gamma = 0.1)
get_initial_theta(y, distname="exp", type="s")

# starting values for the heavy-tailed version of a Normal = Tukey's h
y <- rLambertW(n=1000, beta = c(2, 1), distname = "normal", delta = 0.2)
get_initial_theta(y, distname="normal", type = "h")#'

###
get_theta_bounds(type="hh", distname="normal", beta = c(0, 1))

###
theta.restr <- theta2unbounded(list(beta = c(-1, 0.1),
                                    delta = c(0.2, 0.2)),
                                    distname = "normal")
theta.restr
# returns again the beta and delta from above
theta2unbounded(theta.restr, inverse = TRUE, distname = "normal")
}
\seealso{
\code{\link{check_beta}}

\code{\link{estimate_beta}}, \code{\link{get_initial_tau}}

\code{\link{beta2tau}}
}

