\name{plot_profile}

\alias{plot_profile}

\title{Visualization for Model Selection/Validation}

\description{
  Function for plotting the cross-validated tuning profiles of a \code{PRSP} object. 
  It uses the user's choice of statistics among the Log Hazard Ratio (LHR), 
  Log-Rank Test (LRT) or Concordance Error Rate (CER) as a function of the model tuning parameter, 
  that is, the optimal number of peeling steps of the peeling sequence (inner loop of our PRSP algorithm).
}

\usage{
  plot_profile(object,
               main = NULL, 
               xlab = "Peeling Steps", 
               ylab = "Mean Profiles",
               add.sd = TRUE, 
               add.legend = TRUE, 
               add.profiles = TRUE,
               pch = 20, 
               col = 1, 
               lty = 1, 
               lwd = 2, 
               cex = 2,
               device = NULL, 
               file = "Profile Plot", 
               path=getwd(), 
               horizontal = FALSE, 
               width = 8.5, 
               height = 5.0, ...)
}

\arguments{
  \item{object}{
    Object of class \code{PRSP} as generated by the main function \code{\link[PRIMsrc]{sbh}}.
  }
  \item{main}{
    \code{Character} \code{vector}. Main Title. Defaults to \code{NULL}.
  }
  \item{xlab}{
    \code{Character} \code{vector}. X axis label. Defaults to "Peeling Steps".
  }
  \item{ylab}{
    \code{Character} \code{vector}. Y axis label. Defaults to "Mean Profiles".
  }
  \item{add.sd}{
    \code{Logical} scalar. Shall the standard error bars be plotted? Defaults to \code{TRUE}.
  }
  \item{add.legend}{
    \code{Logical} scalar. Shall the legend be plotted? Defaults to \code{TRUE}.
  }
  \item{add.profiles}{
    \code{Logical} scalar. Shall the individual profiles (for all replicates) be plotted?
    Defaults to \code{TRUE}.
  }
  \item{pch}{
    \code{Integer} scalar of symbol number for all the profiles. Defaults to 20.
  }
  \item{col}{
    \code{Integer} scalar of line color of the mean profile. Defaults to 1.
  }
  \item{lty}{
    \code{Integer} scalar of line type of the mean profile. Defaults to 1.
  }
  \item{lwd}{
    \code{Integer} scalar of line width of the mean profile. Defaults to 2.
  }
  \item{cex}{
    \code{Integer} scalar of symbol expansion for all the profiles. Defaults to 2.
  }
  \item{device}{
    Graphic display device in \{NULL, "PS", "PDF"\}. Defaults to NULL (standard output screen).
    Currently implemented graphic display devices are "PS" (Postscript) or "PDF" (Portable Document Format).
  }
  \item{file}{
    File name for output graphic. Defaults to "Profile Plot".
  }
  \item{path}{
    Absolute path (without final (back)slash separator). Defaults to working directory path.
  }
  \item{horizontal}{
    \code{Logical} scalar. Orientation of the printed image. Defaults to \code{FALSE}, that is potrait orientation.
  }
  \item{width}{
    \code{Numeric} scalar. Width of the graphics region in inches. Defaults to 8.5.
  }
  \item{height}{
    \code{Numeric} scalar. Height of the graphics region in inches. Defaults to 5.
  }
  \item{\dots}{
    Generic arguments passed to other plotting functions.
  }
}

\details{
  Model validation is done by applying the optimization criterion on the user's choice of specific statistic.
  The goal is to find the optimal value of the K-fold cross-validated number of steps by maximization of LHR or LRT, or minimization of CER.
  
  Currently, this done internally for visualization purposes, but it will ultimately offer the option to do be
  interactive with the end-user as well for parameter choosing/model selection.
}

\value{
  Invisible. None. Displays the plot(s) on the specified \code{device}.
}

\references{
    \itemize{
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2015).
              "\emph{Cross-validation and Peeling Strategies for Survival Bump Hunting using Recursive Peeling Methods.}"
              (Submitted).
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2014).
              "\emph{Cross-Validation of Survival Bump Hunting by Recursive Peeling Methods.}"
              In JSM Proceedings, Survival Methods for Risk Estimation/Prediction Section. Boston, MA, USA.
              American Statistical Association IMS - JSM, p. 3366-3380.
        \item Dazard J-E. and J. S. Rao (2010).
              "\emph{Local Sparse Bump Hunting.}"
              J. Comp Graph. Statistics, 19(4):900-92.
    }
}

\author{
    \itemize{
        \item "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}
        \item "Michael Choe, M.D." \email{mjc206@case.edu}
        \item "Michael LeBlanc, Ph.D." \email{mleblanc@fhcrc.org}
        \item "Alberto Santana, MBA." \email{ahs4@case.edu}
    }
    Maintainer: "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}

    Acknowledgments: This project was partially funded by the National Institutes of Health
    NIH - National Cancer Institute (R01-CA160593) to J-E. Dazard and J.S. Rao.
}

\note{
  End-user plotting function.
}

\keyword{Exploratory Survival/Risk Analysis}
\keyword{Survival/Risk Estimation & Prediction}
\keyword{Non-Parametric Method}
\keyword{Cross-Validation}
\keyword{Bump Hunting}
\keyword{Rule-Induction Method}

\examples{
#===================================================
# Loading the library and its dependencies
#===================================================
library("PRIMsrc")

#=================================================================================
# Simulated dataset #1 (n=250, p=3)
# Replicated Combined Cross-Validation (RCCV)
# Peeling criterion = LRT
# Optimization criterion = LRT
#=================================================================================
CVCOMBREP.synt1 <- sbh(dataset = Synthetic.1, 
                       cvtype = "combined", cvcriterion = "lrt",
                       B = 1, K = 5, 
                       vs = TRUE, cpv = FALSE, 
                       decimals = 2, probval = 0.5, 
                       arg = "beta=0.05,
                              alpha=0.1,
                              minn=10,
                              L=NULL,
                              peelcriterion=\"lr\"",
                       parallel = FALSE, conf = NULL, seed = 123)

plot_profile(object = CVCOMBREP.synt1, 
             main = "RCCV tuning profiles for model #1",
             xlab = "Peeling Steps", ylab = "Mean Profiles",
             pch=20, col="black", lty=1, lwd=2, cex=2,
             add.sd = TRUE, add.legend = TRUE, add.profiles = TRUE,
             device = NULL, file = "Profile Plot", path=getwd(),
             horizontal = FALSE, width = 8.5, height = 5)
}
