#' Nonlinearity tests
#' @details
#' This function runs a set of nonlinearity tests on the RR time series 
#' implemented in other R packages including:
#' \itemize{
#'    \item Teraesvirta's neural metwork test for nonlinearity (\code{\link[tseries]{terasvirta.test}}).
#'    \item White neural metwork test for nonlinearity (\code{\link[tseries]{white.test}}).
#'    \item Keenan's one-degree test for nonlinearity (\code{\link[TSA]{Keenan.test}}).
#'    \item Perform the McLeod-Li test for conditional heteroscedascity (ARCH). (\code{\link[TSA]{McLeod.Li.test}}).
#'    \item Perform the Tsay's test for quadratic nonlinearity in a time series. (\code{\link[TSA]{Tsay.test}}).
#'    \item Perform the Likelihood ratio test for threshold nonlinearity. (\code{\link[TSA]{tlrt}}).
#' }
#' @param HRVData Structure containing the RR time series.
#' @param indexNonLinearAnalysis Reference to the data structure that will contain the nonlinear analysis
#' @return A \emph{HRVData} structure containing a \emph{NonlinearityTests} field storing the results of each of the tests. 
#' The \emph{NonlinearityTests} list is stored under the \emph{NonLinearAnalysis} structure.
NonlinearityTests <-function(HRVData, indexNonLinearAnalysis =length(HRVData$NonLinearAnalysis)){
  
  checkingNonLinearIndex(indexNonLinearAnalysis, length(HRVData$NonLinearAnalysis))
  
  if (is.null(HRVData$Beat$RR)){
    stop("RR time series not present\n")
  }
  
  if (HRVData$Verbose){
    cat("  --- Performing nonlinearity tests ---\n")  
  }
  
  tseries = HRVData$Beat$RR
  # apply all tests
  HRVData$NonLinearAnalysis[[indexNonLinearAnalysis]]$NonlinearityTests = 
    nonlinearityTest(time.series = tseries, verbose = HRVData$Verbose)
  
  
  return(HRVData)
  
}

#' Surrogate data testing 
#' @details
#' This function tests the null hypothesis (H0) stating that the series describes a linear process. The
#' test is performed by generating several surrogate data according to H0 and comparing the values
#' of a discriminating statistic between both original data and the surrogate data. If the value of 
#' the statistic is significantly different for the original series than for the surrogate set,
#'  the null hypothesis is rejected and nonlinearity assumed.  The  surrogate data is generated by using a phase randomization procedure.
#' @param HRVData Structure containing the RR time series.
#' @param indexNonLinearAnalysis Reference to the data structure that will contain the nonlinear analysis
#' @param significance Significance of the test
#' @param doPlot Logical value. If TRUE, a graphical representation of the statistic value for
#' both surrogates and original data is shown.
#' @param useFunction The function that computes the discriminating statistic that shall be used for testing.
#' @param ... Additional arguments for the \emph{useFunction} function.
#' @return A \emph{HRVData} structure containing a \emph{SurrogateTest} field storing the 
#' statistics computed for the set (\emph{surrogates.statistics} field) and the RR time series 
#' (\emph{data.statistic field}). The \emph{SurrogateTest} list is stored under the \emph{NonLinearAnalysis} structure.
#' @references SCHREIBER, Thomas; SCHMITZ, Andreas. Surrogate time series. Physica D: 
#' Nonlinear Phenomena, 2000, vol. 142, no 3, p. 346-382.
#' @author Constantino A. Garcia
SurrogateTest <- function(HRVData, 
                          indexNonLinearAnalysis = length(HRVData$NonLinearAnalysis), 
                          significance = 0.05,
                          doPlot = TRUE, useFunction, ...){
  
  checkingNonLinearIndex(indexNonLinearAnalysis, length(HRVData$NonLinearAnalysis))
  
  if (is.null(HRVData$Beat$RR)){
    stop("RR time series not present\n")
  }
          
  tseries = HRVData$Beat$RR
  # apply all tests
  HRVData$NonLinearAnalysis[[indexNonLinearAnalysis]]$SurrogateTest = 
  surrogateTest(time.series = tseries, significance = significance,
                verbose = HRVData$Verbose, 
                do.plot = doPlot, FUN = useFunction, ...)  
  
  return(HRVData)
}
