#' Calculation of proabilities for binary sequences based on the
#' final matrix generated by the genetic algorithm
#'
#' Observation of binary correlated binary data can be expressed as
#' binary sequences. In the case of two binary variables possible
#' observations are (0,0), (0,1), (1,0) and (1,1). In general, 2^m
#' binary sequences are possible, where m is the number of binary
#' variables. Based on the representative matrix generated by the
#' genetic algorithm the probability for each binary sequence is
#' determined.
#' @title Calculation of probabilities for binary sequences
#' @param Xt Representative matrix generated by the genetic algorithm
#'   with \code{\link{iter_matrix}}
#' @return A vector of probabilities for the binary sequences
#' @author Jochen Kruppa, Klaus Jung
#' @references
#' Kruppa, J., Lepenies, B., & Jung, K. (2018). A genetic algorithm for simulating correlated binary data from biomedical research. \emph{Computers in biology and medicine}, \strong{92}, 1-8. \doi{10.1016/j.compbiomed.2017.10.023}
#' @export
#' @examples
#' ### Generation of the representive matrix Xt
#' X0 <- start_matrix(p = c(0.5, 0.6), k = 1000)
#' Xt <- iter_matrix(X0, R = diag(2), T = 10000, e.min = 0.00001)$Xt
#'
#' ### Calculation of probabilities for binary sequences
#' sequence_probs(Xt = Xt)
sequence_probs <- function(Xt) {
  n = dim(Xt)[1]
  m = dim(Xt)[2]
  binseq = rep("free", n)
  for (i in 1:n) {
    binseq_i = "F"
    for (j in 1:m) {
      binseq_i = paste(binseq_i, as.character(Xt[i,j]), sep="")
    }
    binseq[i] = binseq_i
  }
  tabbinseq = table(binseq) / n
  return(tabbinseq)
}
