/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.text;

import java.io.IOException;
import java.math.BigInteger;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.parts.PresentationML.SlidePart;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTBookmark;
import org.docx4j.wml.CTFtnEdnRef;
import org.docx4j.wml.CTMarkupRange;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.P;
import org.docx4j.wml.P.Hyperlink;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.Text;
import org.lysis.reporters.formats.TextProperties;
import org.lysis.reporters.html4r.tools.Format;
import org.lysis.reporters.html4r.tools.utils;
import org.lysis.reporters.pptx4r.elements.Utils;
import org.lysis.reporters.tools.DocComponent;
import org.lysis.reporters.tools.HTML4R;


public class Paragraph extends DocComponent implements HTML4R{
	private LinkedHashMap<Integer, String> hyperlink;
	private LinkedHashMap<Integer, String> textList;
	private LinkedHashMap<Integer, TextProperties> formatList;
	private LinkedHashMap<Integer, FootNote> footnoteList;
	
	private int index;
	
	private boolean hasBookmark;
	private JAXBElement<CTBookmark> bmStart;
	private JAXBElement<CTMarkupRange> bmEnd;
	private boolean vanilla_format;

	public Paragraph( ) {
		textList = new LinkedHashMap<Integer, String>();
		formatList = new LinkedHashMap<Integer, TextProperties>();
		hyperlink = new LinkedHashMap<Integer, String>();
		footnoteList = new LinkedHashMap<Integer, FootNote>();
		index = 0;
		vanilla_format = false;
	}
	
	public void setVanilla(){
		vanilla_format = true;
	}
	
	public void setBookmark(String bookmark, BigInteger bookmarkID){
		hasBookmark = true;
		
		org.docx4j.wml.ObjectFactory factory = Context.getWmlObjectFactory();
		CTBookmark bm = new CTBookmark();
		bm.setId(bookmarkID);
		bm.setName(bookmark);		
		bmStart =  factory.createBodyBookmarkStart(bm);

		CTMarkupRange mr = factory.createCTMarkupRange();
		mr.setId(bookmarkID);
		bmEnd = factory.createBodyBookmarkEnd(mr);
	}
	
	public void addFootnoteToLastEntry( FootNote fn ){
		footnoteList.put(index, fn);
	}
	
	public String asText(){
		String out = "";
		if( index > 0 ){
			for( int i = 1 ; i <= index ; i++ ){
				out += textList.get(i);
			}
		}
		return out;		
	}
	
	public void addText ( String value, TextProperties tp ) throws IOException{
		index++;
		textList.put(index, value );
		formatList.put(index, tp);
		hyperlink.put(index, "");
		footnoteList.put(index, null);
	}
	
	public void addText ( String value ) throws IOException{
		index++;
		textList.put(index, value );
		formatList.put(index, null);
		hyperlink.put(index, "");
		footnoteList.put(index, null);
	}
	
	public void addText ( String value, TextProperties tp, String hlink ) throws IOException{
		index++;
		textList.put(index, value );
		formatList.put(index, tp);
		hyperlink.put(index, hlink);
		footnoteList.put(index, null);
	}
	
	public void addText ( String value, String hlink) throws IOException{
		index++;
		textList.put(index, value );
		formatList.put(index, null);
		hyperlink.put(index, hlink);
		footnoteList.put(index, null);
	}
	
	public void addParagraph ( Paragraph value ) throws IOException{
		value = reindex(value, index );
		textList.putAll(value.textList);
		formatList.putAll(value.formatList);
		hyperlink.putAll(value.hyperlink);
		footnoteList.putAll(value.footnoteList);
		index = index + value.index;
	}
	
	private static Paragraph reindex(Paragraph value, int add){
		if( value.index < 1 ) return value;
		LinkedHashMap<Integer, String> newTextList = new LinkedHashMap<Integer, String>();
		LinkedHashMap<Integer, TextProperties> newFormatList = new LinkedHashMap<Integer, TextProperties>();
		LinkedHashMap<Integer, String> newHyperLinkList = new LinkedHashMap<Integer, String>();
		LinkedHashMap<Integer, FootNote> newFootnoteList = new LinkedHashMap<Integer, FootNote>();
		for( int i = 1 ; i <= value.index ; i++ ){
			newTextList.put( i + add , value.textList.get(i));
			newFormatList.put( i + add , value.formatList.get(i));
			newHyperLinkList.put( i + add , value.hyperlink.get(i));
			newFootnoteList.put( i + add , value.footnoteList.get(i));
		}
		value.textList = newTextList;
		value.formatList = newFormatList;
		value.hyperlink = newHyperLinkList;
		value.footnoteList = newFootnoteList;
		
		return value;
	}
	
	public void setTextProperties(TextProperties tp){
		for (Iterator<Integer> it1 = formatList.keySet().iterator(); it1.hasNext();) {
			int key = it1.next();
			formatList.put(key, tp);
		}
	}
	


	public String toString(){
		String out = "";
		if( index > 0 )
			for( int i = 1 ; i <= index ; i++ ){
				out += textList.get(i);
			}
		return out;
	}

	@Override
	public String getHTML() {
		String out = "";
		if( index > 0 ){
			for( int i = 1 ; i <= index ; i++ ){
				String span = "";
				String[] texts = textList.get(i).split("\n");
				for(int cid = 0 ; cid < texts.length ; cid ++ ){
					if( cid > 0 ) {
						span += "<br>";
					}
					span += texts[cid];
					if( formatList.get(i) != null ){
						TextProperties tp = formatList.get(i);
						String css;
						
						if( vanilla_format )
							css = Format.getJSString(tp.get_html_pr_vanilla());
						else 
							css = Format.getJSString(tp.get_html_pr());
						
						span = utils.span_style( span, css );
					} else span = utils.span( span );
					
				}
				if( !hyperlink.get(i).equals("") ){
					span = utils.a( span, hyperlink.get(i) );
				}
				if( footnoteList.get(i) != null && counters != null){
					footnoteList.get(i).setHTMLReference(list_id, counters, numbDefinition);
					span += utils.tooltip( "", footnoteList.get(i).getHTML() );
				}
				
				out += span;
			}
		}
		return out;
	}

	private void fill_docx_par(P p){
		if( index > 0 ){
			for( int i = 1 ; i <= index ; i++ ){
				
				String[] texts = textList.get(i).split("\n");
				for(int cid = 0 ; cid < texts.length ; cid ++ ){
					R run = new R();
					if( cid > 0 ) {
						Br br = new Br();
						run.getContent().add(br);
					}
					if( formatList.get(i) != null ){
						TextProperties tp = formatList.get(i);
						Text text = new Text();
						text.setValue( texts[cid] );
						text.setSpace("preserve");
						run.getContent().add(text);
						RPr rpr = tp.get_docx_pr();
						run.setRPr(rpr);
						
					} else {
						Text text = new Text();
						text.setValue( texts[cid] );
						text.setSpace("preserve");
						run.getContent().add(text);
					}
					
					if( !hyperlink.get(i).equals("") ){
						ExternalLink extL = new ExternalLink(hyperlink.get(i));
						extL.setWordMLPackage(getDOCXMLPackage());
						Hyperlink hp = null;
						try {
							hp = (Hyperlink)(extL.get_docx_elt());
							hp.getContent().add(run);
							p.getContent().add(hp);
						} catch (Exception e) {
							p.getContent().add(run);
						}
						
					} else {
						p.getContent().add(run);					
					}
				}


				if( hasBookmark ){
					p.getContent().add(0, bmStart);
					p.getContent().add(1, bmEnd);
				}
				
				if( footnoteList.get(i) != null ){
					try {
						footnoteList.get(i).setNumberingDefinitionsPart(ndp);
						footnoteList.get(i).setOrderedNumID(ordered_num_id);
						footnoteList.get(i).setUnorderedNumID(unordered_num_id);
						footnoteList.get(i).setDOCXMLPackage(getDOCXMLPackage());
						footnoteList.get(i).get_docx_elt();
						CTFtnEdnRef ff = new CTFtnEdnRef();
						ff.setId(BigInteger.valueOf(footnoteList.get(i).getFootnoteID()));
						
						ObjectFactory of = new ObjectFactory();

						R runreference = new R();
						runreference.getContent().add(of.createRFootnoteReference(ff));
						runreference.setRPr(footnoteList.get(i).getTextProperties().get_docx_pr());
						p.getContent().add(runreference);

					} catch (Exception e) {
					}
				}
			}
		}
	}
	

	public P getDocxParagraph(String style){
		P p = docx_mlp.getMainDocumentPart().createStyledParagraphOfText(style, "");
		
		fill_docx_par(p);
		return p;
	}
	public P getDocxParagraph(){
		P p = new P();
		fill_docx_par(p);
		return p;
	}


	public CTTextParagraph getPptxTextParagraph(SlidePart slidePart) throws Exception{
		CTTextParagraph textPar = new CTTextParagraph();

		if( index > 0 ){
			List<Object> runs = textPar.getEGTextRun();
			for( int i = 1 ; i <= index ; i++ ){
				CTRegularTextRun textRun = Utils.getRun(textList.get(i));
				runs.add(textRun);

				if( formatList.get(i) != null ){
					TextProperties tp = formatList.get(i);
					CTTextCharacterProperties rpr = tp.get_pptx_pr();
					textRun.setRPr(rpr);
				} else {
					CTTextCharacterProperties rpr = new CTTextCharacterProperties();
					textRun.setRPr(rpr);
				}
				
				if( !hyperlink.get(i).equals("") ){

					ExternalLink extL = new ExternalLink(hyperlink.get(i));

					extL.setSlidePart(slidePart);

					org.docx4j.dml.CTHyperlink hyperlink = (org.docx4j.dml.CTHyperlink)(extL.get_pptx_elt());
					textRun.getRPr().setHlinkClick(hyperlink);
				}
			}
		}
		return textPar;
	}

	
	@Override
	public String getCSS() {
		return "";
	}

	@Override
	public String getJS() {
		return "";
	}

	@Override
	public boolean hasJS() {
		return false;
	}

	@Override
	public boolean hasHTML() {
		return true;
	}

	@Override
	public boolean hasCSS() {
		return false;
	}

}
