%% File Name: IRT.threshold.Rd
%% File Version: 1.42

\name{IRT.threshold}
\alias{IRT.threshold}
\alias{IRT.WrightMap}
\alias{IRT.WrightMap.IRT.threshold}
\alias{print.IRT.threshold}
%- Also NEED an '\alias' for EACH other topic documented here.

\title{
Thurstonian Thresholds and Wright Map for Item Response Models
}

\description{
The function \code{IRT.threshold} computes Thurstonian thresholds
for item response models. It is only based on fitted models
for which the \code{\link{IRT.irfprob}} does exist.

The function \code{IRT.WrightMap} creates a Wright map and works as a wrapper to the
\code{\link[WrightMap:wrightMap]{WrightMap::wrightMap}} function in
the \pkg{WrightMap} package. Wright maps operate
on objects of class \code{\link{IRT.threshold}}.
}




\usage{
IRT.threshold(object, prob.lvl=.5, type="category")

\method{print}{IRT.threshold}(x, \dots)

IRT.WrightMap(object, \dots)

\method{IRT.WrightMap}{IRT.threshold}(object, label.items=NULL, \dots)
}



%- maybe also 'usage' for other objects documented here.

\arguments{
\item{object}{
Object of fitted models for which \code{\link{IRT.irfprob}} exists.
}
\item{prob.lvl}{Requested probability level of thresholds.}
\item{type}{Type of thresholds to be calculated. The default is
    category-wise calculation. If only one threshold per item should
    be calculated, then choose \code{type="item"}. If an item possesses
    a maximum score of \eqn{K_i}, then a threshold is defined as a
    value which produces an expected value of \eqn{K_i /2}
    (see Ali, Chang & Anderson, 2015).}
\item{x}{Object of class \code{IRT.threshold}}
\item{label.items}{Vector of item labels}
\item{\dots}{
Further arguments to be passed.
}
}

%\item{type}{Type of person parameter estimate. \code{"PV"} (plausible values),
%\code{"WLE"} (weighted likelihood estimates) and
%\code{"Pop"} (population trait distribution) can be specified.
%}

%\details{
%A Wright map is only created for models with an assumed normal distibution.
%Hence, not for all models of the \code{\link{tamaan}} functions
%Wright maps are created.
%}


\value{
Function \code{IRT.threshold}:\cr
Matrix with Thurstonian thresholds

Function \code{IRT.WrightMap}: \cr
A Wright map generated by the \pkg{WrightMap} package.
}



\references{
Ali, U. S., Chang, H.-H., & Anderson, C. J. (2015). \emph{Location indices for ordinal
polytomous items based on item response theory} (Research
Report No. RR-15-20). Princeton, NJ: Educational Testing Service.
}

\author{
The \code{IRT.WrightMap} function is based on the
\code{\link[WrightMap:wrightMap]{WrightMap::wrightMap}} function
in the \pkg{\link[WrightMap:WrightMap]{WrightMap}} package.
}

%\note{
%See \url{http://wrightmap.org/post/107431190622/wrightmap-multifaceted-models}
%for creating Wright maps for multi-faceted models.
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
See the \code{\link[WrightMap:wrightMap]{WrightMap::wrightMap}} function in
the \pkg{WrightMap} package.
}

%See \url{http://wrightmap.org/WrightMap} fur tutorials
%of the \pkg{WrightMap} package.

\examples{
\dontrun{
#############################################################################
# EXAMPLE 1: Fitted unidimensional model with gdm
#############################################################################

data(data.Students)
dat <- data.Students

# select part of the dataset
resp <- dat[, paste0("sc",1:4) ]
resp[ paste(resp[,1])==3,1] <-  2
psych::describe(resp)

# Model 1: Partial credit model in gdm
theta.k <- seq( -5, 5, len=21 )   # discretized ability
mod1 <- CDM::gdm( dat=resp, irtmodel="1PL", theta.k=theta.k, skillspace="normal",
              centered.latent=TRUE)

# compute thresholds
thresh1 <- TAM::IRT.threshold(mod1)
print(thresh1)
IRT.WrightMap(thresh1)

#############################################################################
# EXAMPLE 2: Fitted mutidimensional model with gdm
#############################################################################

data( data.fraction2 )
dat <- data.fraction2$data
Qmatrix <- data.fraction2$q.matrix3

# Model 1: 3-dimensional Rasch Model (normal distribution)
theta.k <- seq( -4, 4, len=11 )   # discretized ability
mod1 <- CDM::gdm( dat, irtmodel="1PL", theta.k=theta.k, Qmatrix=Qmatrix,
              centered.latent=TRUE, maxiter=10 )
summary(mod1)

# compute thresholds
thresh1 <- TAM::IRT.threshold(mod1)
print(thresh1)

#############################################################################
# EXAMPLE 3: Item-wise thresholds
#############################################################################

data(data.timssAusTwn.scored)
dat <- data.timssAusTwn.scored
dat <- dat[, grep("M03", colnames(dat) ) ]
summary(dat)

# fit partial credit model
mod <- TAM::tam.mml( dat )
# compute thresholds with tam.threshold function
t1mod <- TAM::tam.threshold( mod )
t1mod
# compute thresholds with IRT.threshold function
t2mod <- TAM::IRT.threshold( mod )
t2mod
# compute item-wise thresholds
t3mod <- TAM::IRT.threshold( mod, type="item")
t3mod
}
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
% \keyword{Wright map}
\keyword{Thurstonian thresholds}
% \keyword{ ~kwd2 }% __ONLY ONE__ keyword per line
