% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xGR2GeneScores.r
\name{xGR2GeneScores}
\alias{xGR2GeneScores}
\title{Function to identify likely modulated seed genes given a list of genomic regions together with the significance level}
\usage{
xGR2GeneScores(data, significance.threshold = 5e-05,
build.conversion = c(NA, "hg38.to.hg19", "hg18.to.hg19"),
distance.max = 50000, decay.kernel = c("slow", "linear", "rapid",
"constant"), decay.exponent = 2, GR.Gene = c("UCSC_knownGene",
"UCSC_knownCanonical"), scoring.scheme = c("max", "sum", "sequential"),
verbose = T, RData.location = "http://galahad.well.ox.ac.uk/bigdata")
}
\arguments{
\item{data}{a named input vector containing the sinificance level for
genomic regions (GR). For this named vector, the element names are GR,
in the format of 'chrN:start-end', where N is either 1-22 or X, start
(or end) is genomic positional number; for example, 'chr1:13-20'. The
element values for the significance level (measured as p-value or fdr).
Alternatively, it can be a matrix or data frame with two columns: 1st
column for GR, 2nd column for the significance level.}

\item{significance.threshold}{the given significance threshold. By
default, it is set to NULL, meaning there is no constraint on the
significance level when transforming the significance level of GR into
scores. If given, those GR below this are considered significant and
thus scored positively. Instead, those above this are considered
insigificant and thus receive no score}

\item{build.conversion}{the conversion from one genome build to
another. The conversions supported are "hg38.to.hg19" and
"hg18.to.hg19". By default it is NA (no need to do so)}

\item{distance.max}{the maximum distance between genes and GR. Only
those genes no far way from this distance will be considered as seed
genes. This parameter will influence the distance-component weights
calculated for nearby GR per gene}

\item{decay.kernel}{a character specifying a decay kernel function. It
can be one of 'slow' for slow decay, 'linear' for linear decay, and
'rapid' for rapid decay. If no distance weight is used, please select
'constant'}

\item{decay.exponent}{a numeric specifying a decay exponent. By
default, it sets to 2}

\item{GR.Gene}{the genomic regions of genes. By default, it is
'UCSC_knownGene', that is, UCSC known genes (together with genomic
locations) based on human genome assembly hg19. It can be
'UCSC_knownCanonical', that is, UCSC known canonical genes (together
with genomic locations) based on human genome assembly hg19.
Alternatively, the user can specify the customised input. To do so,
first save your RData file (containing an GR object) into your local
computer, and make sure the GR object content names refer to Gene
Symbols. Then, tell "GR.Gene" with your RData file name (with or
without extension), plus specify your file RData path in
"RData.location". Note: you can also load your customised GR object
directly}

\item{scoring.scheme}{the method used to calculate seed gene scores
under a set of GR. It can be one of "sum" for adding up, "max" for the
maximum, and "sequential" for the sequential weighting. The sequential
weighting is done via: \eqn{\sum_{i=1}{\frac{R_{i}}{i}}}, where
\eqn{R_{i}} is the \eqn{i^{th}} rank (in a descreasing order)}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to true for display}

\item{RData.location}{the characters to tell the location of built-in
RData files. See \code{\link{xRDataLoader}} for details}
}
\value{
an object of class "mSeed", a list with following components:
\itemize{
\item{\code{GR}: a matrix of nGR X 3 containing GR information, where
nGR is the number of GR, and the 3 columns are "GR" (genomic regions),
"Score" (the scores for GR calculated based on p-values taking into
account the given threshold of the significant level), "Pval" (the
input p-values for GR)}
\item{\code{Gene}: a matrix of nGene X 3 containing Gene information,
where nGene is the number of seed genes, and the 3 columns are "Gene"
(gene symbol), "Score" (the scores for seed genes), "Pval" (pvalue-like
significance level transformed from gene scores)}
\item{\code{call}: the call that produced this result}
}
}
\description{
\code{xGR2GeneScores} is supposed to identify likely modulated seed
genes from a list of SNPs together with the significance level
(measured as p-values or fdr). To do so, it defines seed genes and
their scores that take into account the distance to and the
significance of input SNPs. It returns an object of class "mSeed".
}
\note{
This function uses \code{\link{xGRscores}} and \code{\link{xGR2nGenes}}
to define and score nearby genes that are located within distance
window of input genomic regions.
}
\examples{
\dontrun{
# Load the XGR package and specify the location of built-in data
library(XGR)
RData.location <- "http://galahad.well.ox.ac.uk/bigdata_dev"

# a) provide the seed SNPs with the significance info
## load ImmunoBase
ImmunoBase <- xRDataLoader(RData.customised='ImmunoBase',
RData.location=RData.location)
## get lead SNPs reported in AS GWAS and their significance info (p-values)
gr <- ImmunoBase$AS$variant
df <- as.data.frame(gr, row.names=NULL)
chr <- df$seqnames
start <- df$start
end <- df$end
sig <- df$Pvalue
GR <- paste(chr,':',start,'-',end, sep='')
data <- cbind(GR=GR, Sig=sig)

# b) define and score seed geens
mSeed <- xGR2GeneScores(data=data, RData.location=RData.location)

# c) extract GR info
head(mSeed$GR)

# d) extract gene info
head(mSeed$Gene)
}
}
\seealso{
\code{\link{xGRscores}}, \code{\link{xGR2nGenes}},
\code{\link{xSparseMatrix}}
}

