#' Function to perform connectivity enrichment analysis on the input graph
#'
#' \code{xRWenricher} is supposed to perform connectivity enrichment analysis on the input graph given a list of nodes of interest. The test statistic is the average affinity between the given nodes. The pairwise affinity between two nodes is estimated via short random walks. The null distribution of the test statistic is generated by permuting node labels on the graph (fixed) in a centrality-preserving manner. In brief, all nodes are equally binned by centrality (defined as the mean affinity to all other nodes), and a permuted instance is generated by randomly sampling (a same number of) nodes from the same bin. The connectivity ratio is the observed divided by the expected (the median across the permutations), together with the empirical p-value. 
#'
#' @param data a vector containing node names
#' @param g an object of class "igraph" or "graphNEL". It will be a weighted graph if having an edge attribute 'weight'. The edge directions are ignored for directed graphs
#' @param Amatrix an affinity matrix pre-computed from the input graph. It is symmetric
#' @param num.permutation the number of permutations generating the null distribution
#' @param nbin the number of bins dividing all nodes into the equal number of nodes
#' @param steps an integer specifying the number of steps that random walk performs. By default, it is 4
#' @param chance an integer specifying the chance of remaining at the same vertex. By default, it is 2, the higher the higher chance
#' @param seed an integer specifying the seed
#' @param verbose logical to indicate whether the messages will be displayed in the screen. By default, it sets to true for display
#' @return 
#' a data frame with 9 columns:
#' \itemize{
#'  \item{\code{Obs}: the observed affinity between the given nodes}
#'  \item{\code{Exp}: the expected affinity between the permuted nodes}
#'  \item{\code{std}: the standard deviation of the expected}
#'  \item{\code{fc}: fold change}
#'  \item{\code{zscore}: z-score}
#'  \item{\code{pvalue}: p-value}
#'  \item{\code{or}: a vector containing odds ratio}
#'  \item{\code{CIl}: a vector containing lower bound confidence interval for the odds ratio}
#'  \item{\code{CIu}: a vector containing upper bound confidence interval for the odds ratio}
#' }
#' @note The input graph will treat as an unweighted graph if there is no 'weight' edge attribute associated with. The edge direction is not considered for the purpose of defining pairwise affinity; that is, adjacency matrix and its laplacian version are both symmetric.
#' @export
#' @seealso \code{\link{xRWenricher}}
#' @include xRWenricher.r
#' @examples
#' # 1) generate a random graph according to the ER model
#' set.seed(825)
#' g <- erdos.renyi.game(10, 3/10)
#' V(g)$name <- paste0('n',1:vcount(g))
#'
#' \dontrun{
#' # 2) perform connectivity enrichment analysis
#' data <- V(g)$name[1:3]
#' res <- xRWenricher(data, g, nbin=2)
#' }

xRWenricher <- function(data, g, Amatrix=NULL, num.permutation=2000, nbin=10, steps=4, chance=2, seed=825, verbose=TRUE)
{
    
    startT <- Sys.time()
    if(verbose){
        message(paste(c("Start at ",as.character(startT)), collapse=""), appendLF=TRUE)
        message("", appendLF=TRUE)
    }
    ####################################################################################

    if(class(g)=="graphNEL"){
        ig <- igraph.from.graphNEL(g)
    }else{
        ig <- g
    }
    if (class(ig) != "igraph"){
        stop("The function must apply to either 'igraph' or 'graphNEL' object.\n")
    }
	
	if(igraph::is_directed(ig)){
		ig <- igraph::as.undirected(ig, mode="collapse", edge.attr.comb="max")
	}
	
    ## check mapping between input seed data and graph
    ind <- match(data, V(ig)$name)
    nodes_mapped <- V(ig)$name[ind[!is.na(ind)]]
	
    ## check mapping between input Affinity matrix and graph
    if(!is.null(Amatrix)){
    	ind <- match(V(ig)$name, rownames(Amatrix))
		if(length(ind[!is.na(ind)])!=vcount(ig)){
			warning("The function must require input Affinity matrix (Amatrix) has the same names (both columns and rows) as the input graph.\n")
			Amatrix <- NULL
		}else{
			Amatrix <- Amatrix[ind, ind]
		}
    }
    if(is.null(Amatrix)){
		####################################################################################
		if(verbose){
			now <- Sys.time()
			message(sprintf("\n#######################################################", appendLF=TRUE))
			message(sprintf("'xRWkernel' is being called to pre-computate affinity (%s):", as.character(now)), appendLF=TRUE)
			message(sprintf("#######################################################", appendLF=TRUE))
		}
	
		## generate de novo
    	Amatrix <- xRWkernel(ig, steps=steps, chance=chance, verbose=verbose)
	
		if(verbose){
			now <- Sys.time()
			message(sprintf("#######################################################", appendLF=TRUE))
			message(sprintf("'xRWkernel' has been finished (%s)!", as.character(now)), appendLF=TRUE)
			message(sprintf("#######################################################\n", appendLF=TRUE))
		}
		####################################################################################
    }

	## remove on-diagonal values
	diag(Amatrix) <- 0

    if(verbose){
        message(sprintf("First, calculate the observed affinity between %d given nodes (%s) ...", num.permutation, length(nodes_mapped), as.character(Sys.time())), appendLF=TRUE)
    }
    
	## observed affinity averaged
	ind <- match(nodes_mapped, V(ig)$name)
	y <- as.matrix(Amatrix[ind, ind])
	obs <- mean(y[lower.tri(y,diag=F)])

    if(verbose){
        message(sprintf("Second, permute nodes (%d times) from %d nodes respecting centrality (defined as the mean pairwise affinity to all other nodes) (%s) ...", num.permutation, vcount(ig), as.character(Sys.time())), appendLF=TRUE)
    }

	## equally binned
	vec_rowsum <- Matrix::rowSums(Amatrix)
	breaks <- stats::quantile(vec_rowsum, seq(0, 1, 1/nbin))
	cut_index <- as.numeric(cut(vec_rowsum, breaks=breaks))
	cut_index[is.na(cut_index)] <- 1
	### append onto ig
	V(ig)$bin <- cut_index
    
    ## df_ind_B
	set.seed(seed)
	### per node
	ls_df <- lapply(1:length(nodes_mapped), function(i){
		x <- nodes_mapped[i]
		## all_to_sample:
		ind <- match(x, V(ig)$name)
		all_to_sample <- which(V(ig)$bin == V(ig)$bin[ind])
		## ind_sampled
		ind_sampled <- base::sample(all_to_sample, num.permutation, replace=T)
		res <- data.frame(ind=ind_sampled, B=1:num.permutation, stringsAsFactors=F)
	})
	df_ind_B <- do.call(rbind, ls_df)
	
    if(verbose){
        message(sprintf("\tcalculate centrality per permutation (%s) ...", as.character(Sys.time())), appendLF=TRUE)
    }
    
    ### per B   
	ls_B <- split(x=df_ind_B$ind, f=df_ind_B$B)
	vec_exp <- sapply(ls_B, function(x){
		y <- as.matrix(Amatrix[x, x])
		z <- y[lower.tri(y,diag=F)]
		mean(z)
	})
	
    if(verbose){
        message(sprintf("Third, estimate the significance (%s) ...", as.character(Sys.time())), appendLF=TRUE)
    }
    
	### exp
	exp_mean <- mean(vec_exp)
	exp_std <- stats::sd(vec_exp)
	
	## ratio
	ratio <- obs/exp_mean
	### just in case: obs=0 & exp=0
	ratio[obs==0 & exp_mean==0] <- 1
	###
	
    ## zscore
    zscore <- (obs-exp_mean)/exp_std
    
    ## pvalue
    pvalue <- sum(vec_exp >= obs) / length(vec_exp)
    
	####################
	
    zscore[is.na(zscore)] <- 0
    zscore[is.infinite(zscore)] <- max(zscore[!is.infinite(zscore)])
    pvalue[is.na(ratio)] <- 1
    ratio[is.na(ratio)] <- 1
    
 	####################
 	or <- CIl <- CIu <- NA
 	if(1){
		y <- stats::t.test(vec_exp - obs, alternative="two.sided")
		or <- obs / (y$estimate + obs)
		CIu <- obs / (y$conf.int[1] + obs)
		CIl <- obs / (y$conf.int[2] + obs)
 	}
 	####################
    
	df_enrichment <- data.frame(obs, exp_mean, exp_std, ratio, zscore, pvalue, or, CIl, CIu, row.names=NULL, stringsAsFactors=F)
	colnames(df_enrichment) <- c("Obs", "Exp", "std", "fc", "zscore", "pvalue", "or", "CIl", "CIu")
    
	df_enrichment$zscore <- signif(df_enrichment$zscore, digits=3)
	df_enrichment$pvalue <- signif(df_enrichment$pvalue, digits=2)

	pvals <- signif(df_enrichment$pvalue, digits=2)
	# scientific notations
	df_enrichment$pvalue  <- base::sapply(pvals, function(x){
		if(x < 0.1 & x!=0){
			as.numeric(format(x,scientific=T))
		}else{
			x
		}
	})
    
    ####################################################################################
    endT <- Sys.time()
    if(verbose){
        message(paste(c("\nFinish at ",as.character(endT)), collapse=""), appendLF=TRUE)
    }
    
    runTime <- as.numeric(difftime(strptime(endT, "%Y-%m-%d %H:%M:%S"), strptime(startT, "%Y-%m-%d %H:%M:%S"), units="secs"))
    message(paste(c("Runtime in total is: ",runTime," secs\n"), collapse=""), appendLF=TRUE)

    invisible(df_enrichment)
}


